package de.fhdw.wtf.context.model.collections;

import java.util.Iterator;

import de.fhdw.wtf.context.core.ObjectFactoryProvider;
import de.fhdw.wtf.context.core.TransactionManager;
import de.fhdw.wtf.context.model.Anything;
import de.fhdw.wtf.persistence.meta.Object;
import de.fhdw.wtf.persistence.meta.UnidirectionalLink;
import de.fhdw.wtf.persistence.utils.Tuple;

/**
 * This class provides an Iterator for Persistant* with Links(Unidirectional).
 * 
 * @param <T>
 *            The underlying element type.
 */
public class PersistenceIteratorWithLink<T extends Anything> implements Iterator<T> {
	
	/**
	 * An iterator over the underlying links.
	 */
	private final Iterator<Tuple<UnidirectionalLink, Object>> unidirectionalLinkIterator;
	
	/**
	 * The element read last through the iterator.
	 */
	private UnidirectionalLink lastRead;
	
	/**
	 * True if at least one element has been read through the iterator.
	 */
	private boolean hasLast;
	
	/**
	 * Creates a PersistenceIteratorWithLink.
	 * 
	 * @param unidirectionalLinks
	 *            A collection of the underlying links.
	 */
	public PersistenceIteratorWithLink(final java.util.Collection<Tuple<UnidirectionalLink, Object>> unidirectionalLinks) {
		this.unidirectionalLinkIterator = unidirectionalLinks.iterator();
	}
	
	@Override
	public boolean hasNext() {
		return this.unidirectionalLinkIterator.hasNext();
	}
	
	@Override
	public T next() {
		this.lastRead = this.unidirectionalLinkIterator.next().getFirst();
		this.hasLast = true;
		return this.provideNewInstanceForType(this.lastRead.getTarget());
		
	}
	
	@Override
	public void remove() {
		if (this.hasLast) {
			TransactionManager.getContext().unset(this.lastRead);
		}
		this.unidirectionalLinkIterator.remove();
	}
	
	/**
	 * Instantiates the Java object for the underlying database object.
	 * 
	 * @param object
	 *            The database object.
	 * @return The corresponding Java object.
	 */
	@SuppressWarnings("unchecked")
	private T provideNewInstanceForType(final Object object) {
		return (T) ObjectFactoryProvider.instance().createObject(object);
	}
	
}
