package de.fhdw.wtf.context.core;

import java.math.BigInteger;
import java.util.Collection;
import java.util.HashSet;

import de.fhdw.wtf.persistence.exception.ObjectNotFoundException;
import de.fhdw.wtf.persistence.exception.PersistenceException;
import de.fhdw.wtf.persistence.exception.RuntimePersistenceException;
import de.fhdw.wtf.persistence.facade.ObjectFacade;
import de.fhdw.wtf.persistence.facade.TypeManager;
import de.fhdw.wtf.persistence.meta.Link;
import de.fhdw.wtf.persistence.meta.MapLink;
import de.fhdw.wtf.persistence.meta.Object;
import de.fhdw.wtf.persistence.meta.Transaction;
import de.fhdw.wtf.persistence.meta.UnidirectionalLink;
import de.fhdw.wtf.persistence.meta.UserObject;
import de.fhdw.wtf.persistence.meta.UserType;
import de.fhdw.wtf.persistence.utils.Tuple;

/**
 * A class to represent an intermediate interface to the persistence layer. A context belongs to one long Transaction.
 * It encapsulates database access and is capable of providing a caching mechanism.
 * 
 */
public class PersistenceContext extends ContextWithDatabaseAccess {
	
	/**
	 * The long transaction of this context.
	 */
	public final Transaction transaction;
	
	/**
	 * Creates a new PersistenceContext.
	 * 
	 * @param objectFacade
	 *            The database facade.
	 * @param transaction
	 *            The long transaction of this context.
	 */
	public PersistenceContext(final ObjectFacade objectFacade, final Transaction transaction) {
		super(objectFacade);
		this.transaction = transaction;
	}
	
	@Override
	public Collection<UserObject> find(final String associationName, final String value) {
		try {
			return this.objectFacade.find(
					this.typeManager.getUnidirectionalAssociationForName(associationName),
					value,
					this.transaction);
		} catch (final PersistenceException e) {
			Logger.getInstance().log(e);
			throw new RuntimePersistenceException();
		}
	}
	
	@Override
	public Collection<UserObject> find(final String associationName, final BigInteger value) {
		try {
			return this.objectFacade.find(
					this.typeManager.getUnidirectionalAssociationForName(associationName),
					value,
					this.transaction);
		} catch (final PersistenceException e) {
			Logger.getInstance().log(e);
			throw new RuntimePersistenceException();
		}
	}
	
	@Override
	public Collection<Tuple<UnidirectionalLink, UserObject>> inverseGet(final String associationName,
			final UserObject object) {
		try {
			return this.objectFacade.inverseGet(
					object,
					this.typeManager.getUnidirectionalAssociationForName(associationName),
					this.transaction);
		} catch (final PersistenceException e) {
			Logger.getInstance().log(e);
			throw new RuntimePersistenceException();
		}
	}
	
	@Override
	public Collection<Tuple<UnidirectionalLink, Object>> get(final UserObject object, final String associationName) {
		try {
			return this.objectFacade.get(
					object,
					this.typeManager.getUnidirectionalAssociationForName(associationName),
					this.transaction);
		} catch (final PersistenceException e) {
			Logger.getInstance().log(e);
			throw new RuntimePersistenceException();
		}
	}
	
	@Override
	public Collection<Tuple<MapLink, Object>> get(final UserObject object,
			final String associationName,
			final String key) {
		try {
			return this.objectFacade.get(
					object,
					this.typeManager.getMapAssociationForName(associationName),
					key,
					this.transaction);
		} catch (final ObjectNotFoundException e) {
			return new HashSet<>();
		} catch (final PersistenceException e) {
			Logger.getInstance().log(e);
			throw new RuntimePersistenceException();
		}
	}
	
	@Override
	public Collection<Tuple<MapLink, Object>> get(final UserObject object,
			final String associationName,
			final BigInteger key) {
		try {
			return this.objectFacade.get(
					object,
					this.typeManager.getMapAssociationForName(associationName),
					key,
					this.transaction);
		} catch (final ObjectNotFoundException e) {
			return new HashSet<>();
		} catch (final PersistenceException e) {
			Logger.getInstance().log(e);
			throw new RuntimePersistenceException();
		}
	}
	
	@Override
	public Collection<Tuple<MapLink, Object>> get(final UserObject object,
			final String associationName,
			final UserObject key) {
		try {
			return this.objectFacade.get(
					object,
					this.typeManager.getMapAssociationForName(associationName),
					key,
					this.transaction);
		} catch (final ObjectNotFoundException e) {
			return new HashSet<>();
		} catch (final PersistenceException e) {
			Logger.getInstance().log(e);
			throw new RuntimePersistenceException();
		}
	}
	
	@Override
	public UserObject create(final String typeName) {
		try {
			return this.objectFacade.create((UserType) this.typeManager.getTypeforName(typeName), this.transaction);
		} catch (final PersistenceException e) {
			Logger.getInstance().log(e);
			throw new RuntimePersistenceException();
		}
	}
	
	@Override
	public UnidirectionalLink set(final UserObject object, final String associationName, final String val) {
		try {
			return this.objectFacade.set(
					object,
					this.typeManager.getUnidirectionalAssociationForName(associationName),
					val,
					this.transaction);
		} catch (final PersistenceException e) {
			Logger.getInstance().log(e);
			throw new RuntimePersistenceException();
		}
	}
	
	@Override
	public UnidirectionalLink set(final UserObject object, final String associationName, final BigInteger val) {
		try {
			return this.objectFacade.set(
					object,
					this.typeManager.getUnidirectionalAssociationForName(associationName),
					val,
					this.transaction);
		} catch (final PersistenceException e) {
			Logger.getInstance().log(e);
			throw new RuntimePersistenceException();
		}
	}
	
	@Override
	public UnidirectionalLink set(final UserObject object, final String associationName, final UserObject target) {
		try {
			return this.objectFacade.set(
					object,
					this.typeManager.getUnidirectionalAssociationForName(associationName),
					target,
					this.transaction);
		} catch (final PersistenceException e) {
			Logger.getInstance().log(e);
			throw new RuntimePersistenceException();
		}
	}
	
	@Override
	public MapLink put(final UserObject owner, final String associationName, final UserObject target, final String key) {
		try {
			return this.objectFacade.put(
					owner,
					this.typeManager.getMapAssociationForName(associationName),
					target,
					key,
					this.transaction);
		} catch (final PersistenceException e) {
			Logger.getInstance().log(e);
			throw new RuntimePersistenceException();
		}
	}
	
	@Override
	public MapLink put(final UserObject owner, final String associationName, final BigInteger target, final String key) {
		try {
			return this.objectFacade.put(
					owner,
					this.typeManager.getMapAssociationForName(associationName),
					target,
					key,
					this.transaction);
		} catch (final PersistenceException e) {
			Logger.getInstance().log(e);
			throw new RuntimePersistenceException();
		}
	}
	
	@Override
	public MapLink put(final UserObject owner, final String associationName, final String target, final String key) {
		try {
			return this.objectFacade.put(
					owner,
					this.typeManager.getMapAssociationForName(associationName),
					target,
					key,
					this.transaction);
		} catch (final PersistenceException e) {
			Logger.getInstance().log(e);
			throw new RuntimePersistenceException();
		}
	}
	
	@Override
	public MapLink put(final UserObject owner,
			final String associationName,
			final UserObject target,
			final BigInteger key) {
		try {
			return this.objectFacade.put(
					owner,
					this.typeManager.getMapAssociationForName(associationName),
					target,
					key,
					this.transaction);
		} catch (final PersistenceException e) {
			Logger.getInstance().log(e);
			throw new RuntimePersistenceException();
		}
	}
	
	@Override
	public MapLink put(final UserObject owner,
			final String associationName,
			final BigInteger target,
			final BigInteger key) {
		try {
			return this.objectFacade.put(
					owner,
					this.typeManager.getMapAssociationForName(associationName),
					target,
					key,
					this.transaction);
		} catch (final PersistenceException e) {
			Logger.getInstance().log(e);
			throw new RuntimePersistenceException();
		}
	}
	
	@Override
	public MapLink put(final UserObject owner, final String associationName, final String target, final BigInteger key) {
		try {
			return this.objectFacade.put(
					owner,
					this.typeManager.getMapAssociationForName(associationName),
					target,
					key,
					this.transaction);
		} catch (final PersistenceException e) {
			Logger.getInstance().log(e);
			throw new RuntimePersistenceException();
		}
	}
	
	@Override
	public MapLink put(final UserObject owner,
			final String associationName,
			final UserObject target,
			final UserObject key) {
		try {
			return this.objectFacade.put(
					owner,
					this.typeManager.getMapAssociationForName(associationName),
					target,
					key,
					this.transaction);
		} catch (final PersistenceException e) {
			Logger.getInstance().log(e);
			throw new RuntimePersistenceException();
		}
	}
	
	@Override
	public MapLink put(final UserObject owner,
			final String associationName,
			final BigInteger target,
			final UserObject key) {
		try {
			return this.objectFacade.put(
					owner,
					this.typeManager.getMapAssociationForName(associationName),
					target,
					key,
					this.transaction);
		} catch (final PersistenceException e) {
			Logger.getInstance().log(e);
			throw new RuntimePersistenceException();
		}
	}
	
	@Override
	public MapLink put(final UserObject owner, final String associationName, final String target, final UserObject key) {
		try {
			return this.objectFacade.put(
					owner,
					this.typeManager.getMapAssociationForName(associationName),
					target,
					key,
					this.transaction);
		} catch (final PersistenceException e) {
			Logger.getInstance().log(e);
			throw new RuntimePersistenceException();
		}
	}
	
	@Override
	public void commit() {
		try {
			this.objectFacade.commit(this.transaction);
		} catch (final PersistenceException e) {
			Logger.getInstance().log(e);
			throw new RuntimePersistenceException();
		}
	}
	
	@Override
	public void rollback() {
		try {
			this.objectFacade.rollback(this.transaction);
		} catch (final PersistenceException e) {
			Logger.getInstance().log(e);
			throw new RuntimePersistenceException();
		}
	}
	
	@Override
	public void savePoint() {
		try {
			this.objectFacade.savePoint(this.transaction);
		} catch (final PersistenceException e) {
			Logger.getInstance().log(e);
			throw new RuntimePersistenceException();
		}
	}
	
	@Override
	public void rollbackToSavePoint() {
		try {
			this.objectFacade.rollbackToSavePoint(this.transaction);
		} catch (final PersistenceException e) {
			Logger.getInstance().log(e);
			throw new RuntimePersistenceException();
			
		}
	}
	
	@Override
	public void unset(final Link toUnset) {
		try {
			this.objectFacade.unset(toUnset, this.transaction);
		} catch (final PersistenceException e) {
			Logger.getInstance().log(e);
			throw new RuntimePersistenceException();
		}
	}
	
	@Override
	public TypeManager getTypeManager() {
		return this.objectFacade.getTypeManager();
	}
	
	@Override
	public Collection<UserObject> getObjectsByType(final String typeName) {
		try {
			return this.objectFacade.findAllObjects(
					(UserType) this.typeManager.getTypeforName(typeName),
					this.transaction);
		} catch (final PersistenceException e) {
			Logger.getInstance().log(e);
			throw new RuntimePersistenceException();
		}
	}
	
	@Override
	public UserObject checkout(final long object) {
		try {
			final UserObject result = this.objectFacade.checkUserObjectOut(object, this.transaction);
			return result;
		} catch (final PersistenceException e) {
			Logger.getInstance().log(e);
			throw new RuntimePersistenceException();
		}
	}
	
}
