package de.fhdw.wtf.context.core;

import java.math.BigInteger;
import java.util.Collection;

import de.fhdw.wtf.persistence.facade.TypeManager;
import de.fhdw.wtf.persistence.meta.Link;
import de.fhdw.wtf.persistence.meta.MapLink;
import de.fhdw.wtf.persistence.meta.Object;
import de.fhdw.wtf.persistence.meta.UnidirectionalLink;
import de.fhdw.wtf.persistence.meta.UserObject;
import de.fhdw.wtf.persistence.utils.Tuple;

/**
 * A Context represents the facade for interacting with the persistence layer. The Context is stored in a thread local
 * variable to be accessible to all objects in the current thread. A Context is usually published to the thread by User
 * Access.
 * 
 */
public abstract class Context {
	
	/**
	 * The per-transaction object cache.
	 */
	private final AnyTypeCache cache = new AnyTypeCache();
	
	/**
	 * Returns the object cache.
	 * 
	 * @return The object cache.
	 */
	public AnyTypeCache getCache() {
		return this.cache;
	}
	
	/**
	 * This method provides a Collection of User Object which have an association with a specified name and a given
	 * target value.
	 * 
	 * @param associationName
	 *            The name of the association on which the find will operate.
	 * @param value
	 *            The String value of the association on which the find operates.
	 * @return A Collection of UserObjects which conform to the find query.
	 */
	public abstract Collection<UserObject> find(String associationName, String value);
	
	/**
	 * This method provides a Collection of User Object which have an association with a specified name and a given
	 * target value.
	 * 
	 * @param associationName
	 *            The name of the association on which the find will operate.
	 * @param value
	 *            The Integer value of the association on which the find operates.
	 * @return A Collection of UserObjects which conform to the find query.
	 */
	public abstract Collection<UserObject> find(String associationName, BigInteger value);
	
	/**
	 * This method provides the owner and the UnidirectionalLink, which is an instance of an association with a given
	 * name and which has a specified UserObject as target.
	 * 
	 * @param associationName
	 *            The name of the association.
	 * @param object
	 *            The target object of the association.
	 * @return Provides a Collection of Tuples of Links and UserObject which conform to the search query.
	 */
	public abstract Collection<Tuple<UnidirectionalLink, UserObject>> inverseGet(String associationName,
			UserObject object);
	
	/**
	 * Provides a Collection of Tuples of Links and their target. The provided Links will have the provided object as an
	 * owner and are instances of an association with the provided name.
	 * 
	 * @param object
	 *            The owner of the Links.
	 * @param associationName
	 *            The name of the association which is the type of the links.
	 * @return Provides a Collection of Links.
	 */
	public abstract Collection<Tuple<UnidirectionalLink, Object>> get(UserObject object, String associationName);
	
	/**
	 * Provides a Collection of Tuples of Keyed-Links and their target. The provided Links will have the provided object
	 * as an owner and are instances of an association with the provided name and key.
	 * 
	 * @param object
	 *            The owner of the Links.
	 * @param associationName
	 *            The name of the association which is the type of the links. The target of this link represents a
	 *            MutableMap.
	 * @param key
	 *            The key to get the value for in the map.
	 * @return Provides a Collection of Links.
	 */
	public abstract Collection<Tuple<MapLink, Object>> get(UserObject object, String associationName, String key);
	
	/**
	 * Provides a Collection of Tuples of Keyed-Links and their target. The provided Links will have the provided object
	 * as an owner and are instances of an association with the provided name and key.
	 * 
	 * @param object
	 *            The owner of the Links.
	 * @param associationName
	 *            The name of the association which is the type of the links. The target of this link represents a
	 *            MutableMap.
	 * @param key
	 *            The key to get the value for in the map.
	 * @return Provides a Collection of Links.
	 */
	public abstract Collection<Tuple<MapLink, Object>> get(UserObject object, String associationName, BigInteger key);
	
	/**
	 * Provides a Collection of Tuples of Keyed-Links and their target. The provided Links will have the provided object
	 * as an owner and are instances of an association with the provided name and key.
	 * 
	 * @param object
	 *            The owner of the Links.
	 * @param associationName
	 *            The name of the association which is the type of the links. The target of this link represents a
	 *            MutableMap.
	 * @param key
	 *            The key to get the value for in the map.
	 * @return Provides a Collection of Links.
	 */
	public abstract Collection<Tuple<MapLink, Object>> get(UserObject object, String associationName, UserObject key);
	
	/**
	 * Provides an instance of a Type with a given name.
	 * 
	 * @param typeName
	 *            The name of the UserType.
	 * @return Provides a new User Object.
	 */
	public abstract UserObject create(String typeName);
	
	/**
	 * Creates a new UnidirectionalLink which has the provided object as owner and the provided String as target.
	 * 
	 * @param object
	 *            The owner of the newly created UnidirectionalLink.
	 * @param associationName
	 *            The name of the association which is the Type of the created UnidirectionalLink.
	 * @param val
	 *            The String Value which will be the target of the UnidirectionalLink.
	 * @return A newly created UnidirectionalLink object.
	 */
	public abstract UnidirectionalLink set(UserObject object, String associationName, String val);
	
	/**
	 * Creates a new UnidirectionalLink which has the provided object as owner and the provided Integer as target.
	 * 
	 * @param object
	 *            The owner of the newly created UnidirectionalLink.
	 * @param associationName
	 *            The name of the association which is the Type of the created UnidirectionalLink.
	 * @param val
	 *            The Integer Value which will be the target of the UnidirectionalLink.
	 * @return A newly created UnidirectionalLink object.
	 */
	public abstract UnidirectionalLink set(UserObject object, String associationName, BigInteger val);
	
	/**
	 * Creates a new UnidirectionalLink which has the provided object as owner and the provided User Object as target.
	 * 
	 * @param object
	 *            The owner of the newly created UnidirectionalLink.
	 * @param associationName
	 *            The name of the association which is the Type of the created UnidirectionalLink.
	 * @param target
	 *            The User Object which will be the target of the UnidirectionalLink.
	 * @return A newly created UnidirectionalLink object.
	 */
	public abstract UnidirectionalLink set(UserObject object, String associationName, UserObject target);
	
	/**
	 * Creates a new UnidirectionalLink which has the provided object as owner and another provided object as target.
	 * The target is accessible via the provided key.
	 * 
	 * @param owner
	 *            The owner of the newly created UnidirectionalLink.
	 * @param associationName
	 *            The name of the association which is the Type of the created UnidirectionalLink.
	 * @param target
	 *            The object which will be the target of the UnidirectionalLink.
	 * @param key
	 *            The key to access the target from the owner via the UnidirectionalLink.
	 * @return A newly created UnidirectionalLink object.
	 */
	public abstract MapLink put(UserObject owner, String associationName, UserObject target, String key);
	
	/**
	 * Creates a new UnidirectionalLink which has the provided object as owner and another provided object as target.
	 * The target is accessible via the provided key.
	 * 
	 * @param owner
	 *            The owner of the newly created UnidirectionalLink.
	 * @param associationName
	 *            The name of the association which is the Type of the created UnidirectionalLink.
	 * @param target
	 *            The object which will be the target of the UnidirectionalLink.
	 * @param key
	 *            The key to access the target from the owner via the UnidirectionalLink.
	 * @return A newly created UnidirectionalLink object.
	 */
	public abstract MapLink put(UserObject owner, String associationName, BigInteger target, String key);
	
	/**
	 * Creates a new UnidirectionalLink which has the provided object as owner and another provided object as target.
	 * The target is accessible via the provided key.
	 * 
	 * @param owner
	 *            The owner of the newly created UnidirectionalLink.
	 * @param associationName
	 *            The name of the association which is the Type of the created UnidirectionalLink.
	 * @param target
	 *            The object which will be the target of the UnidirectionalLink.
	 * @param key
	 *            The key to access the target from the owner via the UnidirectionalLink.
	 * @return A newly created UnidirectionalLink object.
	 */
	public abstract MapLink put(UserObject owner, String associationName, String target, String key);
	
	/**
	 * Creates a new UnidirectionalLink which has the provided object as owner and another provided object as target.
	 * The target is accessible via the provided key.
	 * 
	 * @param owner
	 *            The owner of the newly created UnidirectionalLink.
	 * @param associationName
	 *            The name of the association which is the Type of the created UnidirectionalLink.
	 * @param target
	 *            The object which will be the target of the UnidirectionalLink.
	 * @param key
	 *            The key to access the target from the owner via the UnidirectionalLink.
	 * @return A newly created UnidirectionalLink object.
	 */
	public abstract MapLink put(UserObject owner, String associationName, UserObject target, BigInteger key);
	
	/**
	 * Creates a new UnidirectionalLink which has the provided object as owner and another provided object as target.
	 * The target is accessible via the provided key.
	 * 
	 * @param owner
	 *            The owner of the newly created UnidirectionalLink.
	 * @param associationName
	 *            The name of the association which is the Type of the created UnidirectionalLink.
	 * @param target
	 *            The object which will be the target of the UnidirectionalLink.
	 * @param key
	 *            The key to access the target from the owner via the UnidirectionalLink.
	 * @return A newly created UnidirectionalLink object.
	 */
	public abstract MapLink put(UserObject owner, String associationName, BigInteger target, BigInteger key);
	
	/**
	 * Creates a new UnidirectionalLink which has the provided object as owner and another provided object as target.
	 * The target is accessible via the provided key.
	 * 
	 * @param owner
	 *            The owner of the newly created UnidirectionalLink.
	 * @param associationName
	 *            The name of the association which is the Type of the created UnidirectionalLink.
	 * @param target
	 *            The object which will be the target of the UnidirectionalLink.
	 * @param key
	 *            The key to access the target from the owner via the UnidirectionalLink.
	 * @return A newly created UnidirectionalLink object.
	 */
	public abstract MapLink put(UserObject owner, String associationName, String target, BigInteger key);
	
	/**
	 * Creates a new UnidirectionalLink which has the provided object as owner and another provided object as target.
	 * The target is accessible via the provided key.
	 * 
	 * @param owner
	 *            The owner of the newly created UnidirectionalLink.
	 * @param associationName
	 *            The name of the association which is the Type of the created UnidirectionalLink.
	 * @param target
	 *            The object which will be the target of the UnidirectionalLink.
	 * @param key
	 *            The key to access the target from the owner via the UnidirectionalLink.
	 * @return A newly created UnidirectionalLink object.
	 */
	public abstract MapLink put(UserObject owner, String associationName, UserObject target, UserObject key);
	
	/**
	 * Creates a new UnidirectionalLink which has the provided object as owner and another provided object as target.
	 * The target is accessible via the provided key.
	 * 
	 * @param owner
	 *            The owner of the newly created UnidirectionalLink.
	 * @param associationName
	 *            The name of the association which is the Type of the created UnidirectionalLink.
	 * @param target
	 *            The object which will be the target of the UnidirectionalLink.
	 * @param key
	 *            The key to access the target from the owner via the UnidirectionalLink.
	 * @return A newly created UnidirectionalLink object.
	 */
	public abstract MapLink put(UserObject owner, String associationName, BigInteger target, UserObject key);
	
	/**
	 * Creates a new UnidirectionalLink which has the provided object as owner and another provided object as target.
	 * The target is accessible via the provided key.
	 * 
	 * @param owner
	 *            The owner of the newly created UnidirectionalLink.
	 * @param associationName
	 *            The name of the association which is the Type of the created UnidirectionalLink.
	 * @param target
	 *            The object which will be the target of the UnidirectionalLink.
	 * @param key
	 *            The key to access the target from the owner via the UnidirectionalLink.
	 * @return A newly created UnidirectionalLink object.
	 */
	public abstract MapLink put(UserObject owner, String associationName, String target, UserObject key);
	
	/**
	 * This method commits all changes. So they are accessible to all other transaction in the persistence layer It also
	 * closes the current transaction so it can not be used further.
	 */
	public abstract void commit();
	
	/**
	 * This method rolls back all changes of the current transaction. And also closes the actual transaction so it can
	 * not be used anymore.
	 */
	public abstract void rollback();
	
	/**
	 * This method will create a save Point in the persistence layer, so changes will be saved.
	 */
	public abstract void savePoint();
	
	/**
	 * This will roll back all changes did since the last save Point.
	 */
	public abstract void rollbackToSavePoint();
	
	/**
	 * This method removes a given UnidirectionalLink in the persistence Layer.
	 * 
	 * @param toUnset
	 *            The UnidirectionalLink, which will be removed.
	 */
	public abstract void unset(Link toUnset);
	
	/**
	 * Provides the Type Manager, which is used to query Type Information at Runtime.
	 * 
	 * @return An instance of the Type Manager.
	 */
	public abstract TypeManager getTypeManager();
	
	/**
	 * Provides as Collection of User Objects, which contains all instance of a given type.
	 * 
	 * @param typeName
	 *            The name of a Type.
	 * @return A collection of UserObjects.
	 */
	public abstract Collection<UserObject> getObjectsByType(String typeName);
	
	/**
	 * Provides a User Object with the given Id. It also marks the object as read inside the database.
	 * 
	 * @param object
	 *            The Id of the Object.
	 * @return Provides a new User Object with given Id and type.
	 */
	public abstract UserObject checkout(long object);
	
}
