package de.fhdw.wtf.context.model;

import java.util.Collection;
import java.util.Iterator;
import java.util.Vector;

import de.fhdw.wtf.context.core.Observer;
import de.fhdw.wtf.context.core.ObserverEvent;
import de.fhdw.wtf.persistence.meta.UserObject;

/**
 * An abstract superclass to represent any User Type.
 */
public class AnyType implements IAnyType {
	
	/**
	 * Observers registered with this object.
	 */
	private final Collection<Observer> observers = new Vector<>();
	
	/**
	 * Default constructor.
	 */
	public AnyType() {
	}
	
	/**
	 * Loads object from database.
	 * 
	 * @param userObject
	 *            The underlying user object.
	 */
	public AnyType(final UserObject userObject) {
		this.object = userObject;
	}
	
	/**
	 * The associated UserObject.
	 */
	private UserObject object = null;
	
	/**
	 * Sets the associated UserObject. This is done when the object has been freshly created.
	 * 
	 * @param object
	 *            The UserObject to set.
	 */
	@Override
	public void setObject(final UserObject object) {
		this.object = object;
	}
	
	/**
	 * Returns the associated UserObject.
	 * 
	 * @return The associated UserObject.
	 */
	@Override
	public UserObject getObject() {
		return this.object;
	}
	
	/**
	 * Returns the database ID of this object.
	 * 
	 * @return The database ID of this object.
	 */
	public long getId() {
		return this.object.getId();
	}
	
	/**
	 * Registers an observer.
	 * 
	 * @param o
	 *            The observer to register.
	 */
	public void registerObserver(final Observer o) {
		this.observers.add(o);
	}
	
	/**
	 * Unregisters an observer.
	 * 
	 * @param o
	 *            The observer to unregister.
	 */
	public void deregisterObserver(final Observer o) {
		this.observers.remove(o);
	}
	
	/**
	 * Notifies all observers about an event.
	 * 
	 * @param event
	 *            The event.
	 */
	@Override
	public void notifyObservers(final ObserverEvent event) {
		final Iterator<Observer> i = this.observers.iterator();
		while (i.hasNext()) {
			final Observer current = i.next();
			current.update(event);
		}
	}
	
	@Override
	public int hashCode() {
		return (int) this.getId();
	}
	
	@Override
	public boolean equals(final Object obj) {
		if (obj instanceof AnyType) {
			final AnyType other = (AnyType) obj;
			return this.getId() == other.getId();
		}
		return false;
	}
	
}
