package de.fhdw.wtf.context.model;

import java.lang.reflect.Field;
import java.util.Collection;
import java.util.HashSet;

import org.aspectj.lang.annotation.SuppressAjWarnings;
import org.aspectj.lang.reflect.FieldSignature;

import de.fhdw.wtf.context.core.MethodInvokedEvent;
import de.fhdw.wtf.context.core.ObjectFactoryProvider;
import de.fhdw.wtf.context.core.TransactionManager;
import de.fhdw.wtf.context.exception.FrameworkException;
import de.fhdw.wtf.context.model.collections.MutableCollection;
import de.fhdw.wtf.context.model.collections.PersistentListWithLinks;
import de.fhdw.wtf.context.model.collections.functors.DummyCollection;
import de.fhdw.wtf.persistence.meta.IntegerValue;
import de.fhdw.wtf.persistence.meta.Object;
import de.fhdw.wtf.persistence.meta.StringValue;
import de.fhdw.wtf.persistence.meta.UnidirectionalLink;
import de.fhdw.wtf.persistence.meta.UserObject;
import de.fhdw.wtf.persistence.meta.UserTransaction;
import de.fhdw.wtf.persistence.utils.Tuple;

/**
 *	An Aspect which advises all Service classes so they can be used for Persistence Access. 
 *
 */
public aspect ServiceAspectWithDB {
	/**
	 * Contains names of fields which have valid values (i.e. which have either
	 * been written or which have been read from the database).
	 */
	private final transient HashSet<String> Service.valid = new HashSet<String>();
	
	pointcut factoryInit() : execution(AnyType UserObjectFactory+.createSpecificUserType());

	
	/**
	 * Advices the execution of the abstract superclass constructor in Service,
	 * so it can create a User Transaction for this service.
	 */
	after() returning(): execution(Service.new()) {
		final Service result = (Service)thisJoinPoint.getThis();
		final UserTransaction transaction = (UserTransaction)TransactionManager.getContext().create(result.getClass().getName());
		result.setTransaction(transaction);
	}
	
	@SuppressAjWarnings({"adviceDidNotMatch"})
	after(final Service This) returning() : execution(public * (Service+ && ! Service).*(..)) && this(This) {
		This.notifyObservers(new MethodInvokedEvent(thisJoinPointStaticPart.getSignature().toString()));
	}
	
	/**
	 * Advises the Field Access of a UserType, so it uses the persistence Integration.
	 * @param owner The UserObject of the owning object.
	 * @return Provides the reference a User Type.
	 */
	@SuppressAjWarnings({"adviceDidNotMatch"})
	AnyType around(final Service owner) :
		get(!transient AnyType+ Service+.*) && target(owner) {
		final FieldSignature signatur = (FieldSignature)thisJoinPointStaticPart.getSignature();
		final String signatureString = this.stripSignature(signatur.toString());
		if (owner.valid.contains(signatureString)) {
			return proceed(owner);
		}
		final Collection<Tuple<UnidirectionalLink, Object>> databaseResults = TransactionManager.getContext().get(owner.getObject(), signatureString);
		owner.valid.add(signatureString);
		if (databaseResults.isEmpty()) {
			return null;
		}
		final UserObject object = (UserObject) databaseResults.iterator().next().getSecond();
		final AnyType result = (AnyType) ObjectFactoryProvider.instance().createObject(object);
		final Field field = signatur.getField();
		field.setAccessible(true);
		try {
			field.set(owner, result);
			owner.valid.add(signatureString);
		}
		catch (final IllegalAccessException e) {
			throw new FrameworkException("Field access error : "+e.getMessage());
		}
		return result;
	}
	
	/**
	 * Advises the Field Access of a Str value in a service, so it uses the persistence Integration.
	 * @param owner The User Object of the owning object.
	 * @return Provides a String, which the value of this field.
	 */
	@SuppressAjWarnings({"adviceDidNotMatch"})
	Str around(final Service owner) :
		get(!transient Str Service+.*) && target(owner) {
		final FieldSignature signatur = (FieldSignature)thisJoinPointStaticPart.getSignature();
		final String signatureString = this.stripSignature(signatur.toString());
		if (owner.valid.contains(signatureString)) {
			return proceed(owner);
		}
		final Str result =  new Str((StringValue)TransactionManager.getContext().get(owner.getObject(), signatureString).iterator().next().getSecond());
		owner.valid.add(signatureString);
		final Field field = signatur.getField();
		field.setAccessible(true);
		try {
			field.set(owner, result);
			owner.valid.add(signatureString);
		}
		catch (final IllegalAccessException e) {
			throw new FrameworkException("Field access error : "+e.getMessage());
		}
		return result;
	}
	
	/**
	 * Advises the Access of a Int Field in a Service Class so it can use persistence Access.
	 * @param owner The User Object of the owning object.
	 * @return Provides an Int which is the value of the field.
	 */
	@SuppressAjWarnings({"adviceDidNotMatch"})
	Int around(final Service owner) :
		get(!transient Int Service+.*) && target(owner) {
		final FieldSignature signatur = (FieldSignature)thisJoinPointStaticPart.getSignature();
		final String signatureString = this.stripSignature(signatur.toString());
		if (owner.valid.contains(signatureString)) {
			return proceed(owner);
		}
		final Int result =  new Int((IntegerValue)TransactionManager.getContext().get(owner.getObject(), signatureString).iterator().next().getSecond());
		owner.valid.add(signatureString);
		final Field field = signatur.getField();
		field.setAccessible(true);
		try {
			field.set(owner, result);
			owner.valid.add(signatureString);
		}
		catch (final IllegalAccessException e) {
			throw new FrameworkException("Field access error : "+e.getMessage());
		}
		return result;
	}
	
	
	@SuppressAjWarnings({"adviceDidNotMatch"})
	after(final Service owner, final Str value) :
		set(!transient Str Service+.*) && target(owner) && args(value)  {
		final String signatur = this.stripSignature(thisJoinPointStaticPart.toString());
		owner.valid.add(signatur);
		TransactionManager.getContext().set(owner.getObject(), signatur, value.toString());
	}
	
	@SuppressAjWarnings({"adviceDidNotMatch"})
	after(final Service owner, final Int value) :
		set(!transient Int Service+.*) && target(owner) && args(value)  {
		final String signatur = this.stripSignature(thisJoinPointStaticPart.toString());
		owner.valid.add(signatur);
		TransactionManager.getContext().set(owner.getObject(), signatur, value.getVal());
	}
	
	 @SuppressAjWarnings({"adviceDidNotMatch"})
	after(final Service owner, final AnyType target) :
		set(!transient Int Service+.*) && target(owner) && args(target)  {
		final String signatur = this.stripSignature(thisJoinPointStaticPart.toString());
		owner.valid.add(signatur);
		TransactionManager.getContext().set(owner.getObject(), signatur, target.getObject());
	}
	
	@SuppressAjWarnings({"adviceDidNotMatch"})
	@SuppressWarnings({ "rawtypes", "unchecked" })
	MutableCollection around(final Service owner) : get(!transient MutableCollection Service+.*)&& target(owner) && !cflow(factoryInit()){
		final FieldSignature signatur = (FieldSignature)thisJoinPointStaticPart.getSignature();
		final String signatureString = this.stripSignature(signatur.toString());
		final MutableCollection result = new PersistentListWithLinks(TransactionManager.getContext().get(owner.getObject(), signatureString), signatureString, owner.getObject());
		return result;
	}
	
	@SuppressAjWarnings({"adviceDidNotMatch"})
	MutableCollection around(final Service owner) : get(!transient MutableCollection Service+.*)&& target(owner) && cflow(factoryInit()) {
		return new DummyCollection<>();
	}
	
	/**
	 * A method to remove the trailing Type of an association provided by the Aspect J Field Information.
	 * @param sig A Field signature which will be stripped.
	 * @return Provides the stripped signature.
	 */
	private String stripSignature(final String sig) {
		String result = sig.substring(sig.indexOf(' ')+1);
		if (result.charAt(result.length()-1) == ')') {
			result = result.substring(0,result.length()-1);
		}
		return result;
	}

}
