package de.fhdw.wtf.context.model;

import de.fhdw.wtf.persistence.meta.StringValue;

/**
 * A class to represent a sequence of Unicode-Characters with the maximum length of 4000 characters.
 * 
 */
public final class Str extends AnyValue {
	
	/**
	 * The representation of Str inside the JVM.
	 */
	private final StringBuilder value;
	
	/**
	 * Private standard constructor needing the value as a java.lang.StringBuilder.
	 * 
	 * @param value
	 *            The string value.
	 */
	private Str(final StringBuilder value) {
		this.value = value;
	}
	
	/**
	 * Creates a new Str object from a StringValue.
	 * 
	 * @param value
	 *            The string value.
	 */
	public Str(final StringValue value) {
		this(value.getValue());
	}
	
	/**
	 * Creates a new Str object from the java.lang.String representation.
	 * 
	 * @param value
	 *            The string value.
	 */
	public Str(final String value) {
		this(new StringBuilder(value));
	}
	
	/**
	 * Creates an empty Str object.
	 */
	public Str() {
		this.value = new StringBuilder();
	}
	
	/**
	 * Performs the concatenation of two Strs with each other.
	 * 
	 * @param tail
	 *            Another Str which will be prepended.
	 * @return Provides a new Str with the format (this)(tail).
	 */
	public Str concat(final Str tail) {
		final StringBuilder nw = new StringBuilder(this.value);
		nw.append(tail.value.substring(0));
		return new Str(nw);
	}
	
	/**
	 * Procides a sub sequence of this Str which starts at poistion from and end before position to. Please note, that
	 * an array out of bound exception might occur if to is larger than the length of this Str.
	 * 
	 * @param from
	 *            The starting position of the sub sequence.
	 * @param to
	 *            The ending position of the sub sequence.
	 * @return The rquested sub sequence.
	 */
	public Str substring(final int from, final int to) {
		final StringBuilder res = new StringBuilder(this.value.substring(from, to));
		return new Str(res);
	}
	
	/**
	 * Compares to this Str to another Str. An checks if this is lexigographical less or equal to another.
	 * 
	 * @param other
	 *            Another Str.
	 * @return Provides true if this is lexigographical less than the other.
	 */
	public boolean lessEq(final Str other) {
		return this.value.substring(0).compareTo(other.value.substring(0)) <= 0;
	}
	
	@Override
	public boolean equals(final Object obj) {
		if (obj instanceof Str) {
			final Str other = (Str) obj;
			return this.value.toString().equals(other.value.toString());
		}
		return false;
	}
	
	@Override
	public int hashCode() {
		return this.value.hashCode();
	}
	
	@Override
	public String toString() {
		return this.value.toString();
	}
	
}
