package de.fhdw.wtf.context.core;

import java.util.Collection;
import java.util.Set;
import java.util.Vector;

import de.fhdw.wtf.context.exception.FrameworkException;
import de.fhdw.wtf.persistence.exception.PersistenceException;
import de.fhdw.wtf.persistence.exception.RuntimePersistenceException;
import de.fhdw.wtf.persistence.facade.AuthorizationFacade;
import de.fhdw.wtf.persistence.meta.Mandant;
import de.fhdw.wtf.persistence.meta.Role;
import de.fhdw.wtf.persistence.meta.User;

/**
 * Login service which can be used to create users and roles.
 */
public class LoginService {
	
	/**
	 * The currently logged on root user. If null, no root user is logged on.
	 */
	private Mandant currentMandant;
	
	/**
	 * The underlying authorization facade.
	 */
	private final AuthorizationFacade facade;
	
	/**
	 * The available roles.
	 */
	private Collection<Role> availableRoles;
	
	/**
	 * Constructor. Reads all available roles.
	 * 
	 * @param facade
	 *            The underlying authorization facade.
	 */
	public LoginService(final AuthorizationFacade facade) {
		this.facade = facade;
		try {
			this.availableRoles = facade.listAllRoles();
		} catch (final PersistenceException e) {
			this.availableRoles = new Vector<>();
			e.printStackTrace();
		}
	}
	
	/**
	 * Retrieve a list containing all users known to the system.
	 * 
	 * @return All known users.
	 */
	public Collection<User> getUsers() {
		this.checkRootLogin();
		try {
			return this.facade.getUsers();
		} catch (final PersistenceException e) {
			throw new RuntimePersistenceException(e);
		}
	}
	
	/**
	 * Checks if there is a root user logged on.
	 * 
	 * @throws FrameworkException
	 *             if no root user is logged on.
	 */
	private void checkRootLogin() {
		if (this.currentMandant == null) {
			throw new FrameworkException("Root not logged in");
		}
	}
	
	/**
	 * Grants a role to a user.
	 * 
	 * @param user
	 *            The user.
	 * @param role
	 *            The role to grant.
	 */
	public void addRole(final User user, final Role role) {
		this.checkRootLogin();
		try {
			this.facade.grantRoleToUser(user, role);
		} catch (final PersistenceException e) {
			throw new RuntimePersistenceException(e);
		}
	}
	
	/**
	 * Creates a new user.
	 * 
	 * @param username
	 *            The name of the user.
	 * @param password
	 *            The password of the user.
	 * @param passwordConfirmation
	 *            The password of the user (a second time).
	 * @return A User object.
	 * @throws FrameworkException
	 *             if the passwords do not match.
	 */
	public User createUser(final String username, final String password, final String passwordConfirmation) {
		this.checkRootLogin();
		if (!password.equals(passwordConfirmation)) {
			throw new FrameworkException("You mispelled your password once");
		}
		try {
			return this.facade.createUser(username, password);
		} catch (final PersistenceException e) {
			throw new RuntimePersistenceException(e);
		}
	}
	
	/**
	 * Returns the roles fetched at construction time.
	 * 
	 * @return All known roles.
	 */
	public Collection<Role> getRoles() {
		this.checkRootLogin();
		return this.availableRoles;
	}
	
	/**
	 * Checks whether the password for a given user is correct.
	 * 
	 * @param mandant
	 *            The root user.
	 * @param username
	 *            The name of the user.
	 * @param password
	 *            The password of the user.
	 * @return True if password is correct, else false.
	 */
	public boolean isLoginCorrect(final String mandant, final String username, final String password) {
		try {
			return this.facade.isPasswordCorrect(mandant, username, password);
		} catch (final PersistenceException e) {
			throw new RuntimePersistenceException(e);
		}
	}
	
	/**
	 * Lets a root user log on.
	 * 
	 * @param mandant
	 *            The name of the root user.
	 * @param password
	 *            The password of the root user.
	 */
	public void rootLogin(final String mandant, final String password) {
		final Mandant m = new Mandant(mandant, password);
		try {
			this.facade.logIn(m);
			this.currentMandant = m;
		} catch (final PersistenceException e) {
			throw new RuntimePersistenceException(e);
		}
	}
	
	/**
	 * Returns all roles for a given user.
	 * 
	 * @param user
	 *            The user whose roles to query.
	 * @return The roles of the given user.
	 */
	public Set<Role> getRolesFrom(final User user) {
		try {
			return this.facade.getRolesFrom(user);
		} catch (final PersistenceException e) {
			throw new RuntimePersistenceException(e);
		}
	}
	
	/**
	 * Logs off the current root user.
	 */
	public void logOff() {
		this.currentMandant = null;
	}
	
}
