package de.fhdw.wtf.context.core;

import java.util.HashMap;
import java.util.Map;

import de.fhdw.wtf.context.model.Anything;
import de.fhdw.wtf.context.model.UserObjectFactory;
import de.fhdw.wtf.persistence.exception.RuntimePersistenceException;
import de.fhdw.wtf.persistence.exception.TypeOrAssociationNotFoundException;
import de.fhdw.wtf.persistence.facade.TypeManager;
import de.fhdw.wtf.persistence.meta.IntegerType;
import de.fhdw.wtf.persistence.meta.Object;
import de.fhdw.wtf.persistence.meta.StringType;

/**
 * Manages object factories.
 */
public final class ObjectFactoryProvider {
	
	/**
	 * The object factories.
	 */
	private final Map<Long, ObjectFactory> factories;
	
	/**
	 * The singleton instance of this class.
	 */
	private static ObjectFactoryProvider instance = null;
	
	/**
	 * Creates an ObjectFactoryProvider.
	 */
	private ObjectFactoryProvider() {
		this.factories = new HashMap<>();
		this.factories.put(IntegerType.Integer_ID, IntegerFactory.create());
		this.factories.put(StringType.String_ID, StringFactory.instance());
	}
	
	/**
	 * Registers an ObjectFactory.
	 * 
	 * @param typeManager
	 *            The TypeManager to use in order to look up the class name.
	 * @param fullQualifiedClassName
	 *            The fully qualified class name to create objects of.
	 * @param factory
	 *            The factory.
	 * @throws TypeOrAssociationNotFoundException
	 *             if the class name is not known by the TypeManager.
	 */
	public void registerTypeFactory(final TypeManager typeManager,
			final String fullQualifiedClassName,
			final UserObjectFactory factory) throws TypeOrAssociationNotFoundException {
		this.factories.put(typeManager.getTypeforName(fullQualifiedClassName).getId(), factory);
		
	}
	
	/**
	 * Returns the singleton instance of this class.
	 * 
	 * @return The singleton instance of this class.
	 */
	public static synchronized ObjectFactoryProvider instance() {
		if (instance == null) {
			instance = new ObjectFactoryProvider();
		}
		return instance;
	}
	
	/**
	 * Creates a suitable Java object given a database object by looking up the corresponding ObjectFactory and invoking
	 * {@link ObjectFactory#createObject} on it.
	 * 
	 * @param object
	 *            The database object.
	 * @return The corresponding Java object.
	 */
	public Anything createObject(final Object object) {
		if (!this.factories.containsKey(object.getInstanceOf().getId())) {
			throw new RuntimePersistenceException();
		}
		return this.factories.get(object.getInstanceOf().getId()).createObject(object);
	}
}
