package de.fhdw.wtf.context.model.collections;

import java.util.Iterator;
import java.util.Vector;

import de.fhdw.wtf.context.exception.FrameworkException;
import de.fhdw.wtf.context.model.Anything;
import de.fhdw.wtf.context.model.collections.functors.Predicate;
import de.fhdw.wtf.context.model.collections.functors.Procedure;

/**
 * A class to represent a list of values in no-db context.
 * 
 * @param <T>
 *            any subtype of anything.
 */
public class TransientList<T extends Anything> extends MutableList<T> {
	
	/**
	 * The Elements in the Collection.
	 */
	private final java.util.List<T> elements;
	
	/**
	 * The constructor for a new Collection.
	 */
	public TransientList() {
		this.elements = new Vector<>();
	}
	
	@Override
	public Iterator<T> iterator() {
		return this.elements.iterator();
	}
	
	@Override
	public Collection<T> union(final Collection<? extends T> otherCollection) {
		if (otherCollection instanceof ImmutableCollection) {
			this.addImmutableCollection((ImmutableCollection<? extends T>) otherCollection);
		} else if (otherCollection instanceof MutableCollection) {
			this.addMutableCollection((MutableCollection<? extends T>) otherCollection);
		} else {
			throw new FrameworkException("Collection type is not supported");
		}
		return this;
	}
	
	/**
	 * Adds the elements of a MutableCollection to this list.
	 * 
	 * @param otherCollection
	 *            The MutableCollection.
	 */
	private void addMutableCollection(final MutableCollection<? extends T> otherCollection) {
		final Iterator<? extends T> iterator = otherCollection.iterator();
		while (iterator.hasNext()) {
			final T current = iterator.next();
			this.elements.add(current);
		}
	}
	
	/**
	 * Adds the elements of an ImmutableCollection to this list.
	 * 
	 * @param otherCollection
	 *            The ImmutableCollection.
	 */
	private void addImmutableCollection(final ImmutableCollection<? extends T> otherCollection) {
		if (!otherCollection.isEmpty()) {
			this.elements.add(otherCollection.front());
			this.addImmutableCollection(otherCollection.tail());
		}
	}
	
	@Override
	public boolean contains(final T element) {
		return this.elements.contains(element);
	}
	
	@Override
	public void insert(final T element) {
		this.elements.add(element);
	}
	
	@Override
	public void apply(final Procedure<T> procedure) {
		final Iterator<T> iterator = this.iterator();
		while (iterator.hasNext()) {
			final T current = iterator.next();
			procedure.op(current);
		}
	}
	
	@Override
	public void remove(final Predicate<T> predicate) {
		final Iterator<T> iterator = this.iterator();
		while (iterator.hasNext()) {
			final T current = iterator.next();
			if (predicate.p(current)) {
				iterator.remove();
			}
		}
	}
	
	@Override
	public boolean isEmpty() {
		return this.elements.isEmpty();
	}
	
	@Override
	public ImmutableCollection<T> copy() {
		ImmutableCollection<T> result = new ImmutableList<>();
		final Iterator<T> iterator = this.elements.iterator();
		while (iterator.hasNext()) {
			final T current = iterator.next();
			result = result.add(current);
		}
		return result;
	}
	
	@Override
	public boolean equals(final Object obj) {
		if (obj instanceof TransientList<?>) {
			final TransientList<?> other = (TransientList<?>) obj;
			return this.elements.equals(other.elements);
		}
		if (obj instanceof MutableCollection<?>) {
			try {
				@SuppressWarnings("unchecked")
				final MutableCollection<T> other = (MutableCollection<T>) obj;
				final Iterator<T> thisIterator = this.iterator();
				while (thisIterator.hasNext()) {
					final T current = thisIterator.next();
					if (!other.contains(current)) {
						return false;
					}
				}
				final Iterator<T> otherIterator = other.iterator();
				while (otherIterator.hasNext()) {
					final T current = otherIterator.next();
					if (!this.contains(current)) {
						return false;
					}
				}
				return true;
			} catch (final ClassCastException e) {
				return false;
			}
		}
		return false;
	}
	
	@Override
	public int hashCode() {
		return this.elements.hashCode();
	}
	
	@Override
	public String toString() {
		return this.elements.toString();
	}
	
}
