package de.fhdw.wtf.persistence.facade;

import java.util.Collection;

import de.fhdw.wtf.persistence.exception.TypeOrAssociationNotFoundException;
import de.fhdw.wtf.persistence.meta.Association;
import de.fhdw.wtf.persistence.meta.MapAssociation;
import de.fhdw.wtf.persistence.meta.Specialization;
import de.fhdw.wtf.persistence.meta.Type;
import de.fhdw.wtf.persistence.meta.UnidirectionalAssociation;

/**
 * Interface of a Type Manager. A Type Manager stores information about all Model Items created by a Class Facade.
 * 
 */
public interface TypeManager {
	
	/**
	 * Stores the Information of a type in this Type Manager Object.
	 * 
	 * @param type
	 *            The Type which information will be saved.
	 */
	void saveType(Type type);
	
	/**
	 * Stores the Information of an association in this Type Manager Object.
	 * 
	 * @param association
	 *            The association, which information will be saved.
	 */
	void saveAssociation(Association association);
	
	/**
	 * Stores the Information of an specialization in this Type Manager Object.
	 * 
	 * @param specialization
	 *            The specialization, which information will be saved.
	 */
	void saveSpecialization(Specialization specialization);
	
	/**
	 * @return Returns a copy of the collection with all specializations saved in this TypeManager. However the
	 *         contained specialization-objects are not copies!
	 */
	Collection<Specialization> getAllSpecializations();
	
	/**
	 * get the last type id.
	 * 
	 * @return the last id
	 */
	long getMaximumTypeId();
	
	/**
	 * Provides the Type for a given ID.
	 * 
	 * @param id
	 *            The id of the searched Type
	 * @return Provides a Type Object.
	 * @throws TypeOrAssociationNotFoundException
	 *             This exception is thrown if there is no Type with the given Id.
	 */
	Type getTypeForId(long id) throws TypeOrAssociationNotFoundException;
	
	/**
	 * Provides the Type for a given name. Note that a name might be ambigous.
	 * 
	 * @param name
	 *            the name of the searched Type.
	 * @return Provides a Type object with the given name.
	 * @throws TypeOrAssociationNotFoundException
	 *             This exception is thrown if there is no Type with the given name.
	 */
	Type getTypeforName(String name) throws TypeOrAssociationNotFoundException;
	
	/**
	 * Provides the UnidirectionalAssociation with the given Id.
	 * 
	 * @param id
	 *            The Id of the searched association.
	 * @return Provides an UnidirectionalAssociation Object with the given name.
	 * @throws TypeOrAssociationNotFoundException
	 *             This exception is thrown if there is no UnidirectionalAssociation with the given Id.
	 */
	UnidirectionalAssociation getUnidirectionalAssociationForId(long id) throws TypeOrAssociationNotFoundException;
	
	/**
	 * Provides the MapAssociation with the given Id.
	 * 
	 * @param id
	 *            The Id of the searched association.
	 * @return Provides an UnidirectionalAssociation Object with the given name.
	 * @throws TypeOrAssociationNotFoundException
	 *             This exception is thrown if there is no MapAssociation with the given Id.
	 */
	MapAssociation getMapAssociationForId(long id) throws TypeOrAssociationNotFoundException;
	
	/**
	 * Provides the UnidirectionalAssociation with the given Name. Note that the name might be ambiguous.
	 * 
	 * @param name
	 *            The Name of the searched UnidirectionalAssociation.
	 * @return Provides an UnidirectionalAssociation Object with the given Name.
	 * @throws TypeOrAssociationNotFoundException
	 *             This exception is thrown if there is no UnidirectionalAssociation with the given name.
	 */
	UnidirectionalAssociation getUnidirectionalAssociationForName(String name)
			throws TypeOrAssociationNotFoundException;
	
	/**
	 * Provides the MapAssociation with the given Name. Note that the name might be ambiguous.
	 * 
	 * @param name
	 *            The Name of the searched UnidirectionalAssociation.
	 * @return Provides an UnidirectionalAssociation Object with the given Name.
	 * @throws TypeOrAssociationNotFoundException
	 *             This exception is thrown if there is no MapAssociation with the given name.
	 */
	MapAssociation getMapAssociationForName(String name) throws TypeOrAssociationNotFoundException;
	
	/**
	 * Deletes all Elements which have been saved by this TypeManager.
	 */
	void clear();
	
	/**
	 * Deletes a type with the given id from this TypeManager.
	 * 
	 * @param typeId
	 *            the id of the UserType to delete.
	 * @throws TypeOrAssociationNotFoundException
	 *             if no UserType with the given id exists.
	 */
	void deleteType(long typeId) throws TypeOrAssociationNotFoundException;
	
	/**
	 * Deletes an association with the given id from this TypeManager.
	 * 
	 * @param associationId
	 *            the id of the Association to delete.
	 * @throws TypeOrAssociationNotFoundException
	 *             if no Association with the given id exists.
	 */
	void deleteAssociation(long associationId) throws TypeOrAssociationNotFoundException;
	
	/**
	 * Deletes the given Specialization from this TypeManager. Does nothing if the given Specialization does not exist
	 * in this TypeManager.
	 * 
	 * @param specialization
	 *            the Specialization to be deleted.
	 */
	void deleteSpecialization(Specialization specialization);
	
	/**
	 * Checks if there is a Type with the given ID.
	 * 
	 * @param typeId
	 *            the id that will be checked
	 * @return true = Type exists, false = Type not exists
	 */
	boolean existsType(long typeId);
	
	/**
	 * Checks if there is a Type with the given Name.
	 * 
	 * @param typeName
	 *            the Name that will be checked
	 * @return true = Type exists, false = Type not exists
	 */
	boolean existsType(String typeName);
	
	/**
	 * Checks if there is a Type with the given ID.
	 * 
	 * @param associationId
	 *            the id that will be checked
	 * @return true = Association exists, false = Association not exists
	 */
	boolean existsAssociation(long associationId);
	
	/**
	 * Checks if there is a Specialization with the given ID.
	 * 
	 * @param specializationId
	 *            the id that will be checked
	 * @return true = Specialization exists, false = Specialization not exists
	 */
	boolean existsSpecialization(long specializationId);
}
