package de.fhdw.wtf.persistence.facade;

import java.sql.CallableStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Vector;

import oracle.jdbc.OracleCallableStatement;
import oracle.jdbc.OracleTypes;
import oracle.sql.ARRAY;
import oracle.sql.ArrayDescriptor;
import de.fhdw.wtf.persistence.exception.ClassFacadeUninitializedException;
import de.fhdw.wtf.persistence.exception.InitializingDatabaseContractViolationException;
import de.fhdw.wtf.persistence.exception.OtherSQLException;
import de.fhdw.wtf.persistence.exception.PersistenceException;
import de.fhdw.wtf.persistence.exception.SpecializationCycleDetected;
import de.fhdw.wtf.persistence.meta.Association;
import de.fhdw.wtf.persistence.meta.IntegerType;
import de.fhdw.wtf.persistence.meta.MapAssociation;
import de.fhdw.wtf.persistence.meta.StringType;
import de.fhdw.wtf.persistence.meta.Type;
import de.fhdw.wtf.persistence.meta.UnidirectionalAssociation;
import de.fhdw.wtf.persistence.meta.UserType;
import de.fhdw.wtf.persistence.utils.IntegerConstants;

/**
 * A class to represent an implementation of the ClassFacade Interface for the Oracle Database. Supports a Database
 * Initializer.
 */
public class OracleClassFacadeImplementation implements ClassFacade {
	
	/**
	 * A flag to determine, if the ClassFacade was initialized,e.g. TypeManager is filled with information and base
	 * types are initialized.
	 */
	private boolean initialized;
	
	/**
	 * The Type Manager which stores the Model Item informations.
	 */
	private TypeManagerImplementation typeManager;
	
	/**
	 * The Oracle Database Manager, which is needed to call the Database.
	 */
	private final OracleDatabaseManager database;
	
	/**
	 * Constructor for a new OracleClassFacadeImplementation.
	 * 
	 * @param database
	 *            The OracleDatabaseManager, which has to be connected.
	 */
	public OracleClassFacadeImplementation(final OracleDatabaseManager database) {
		this.database = database;
		this.initialized = false;
		this.typeManager = TypeManagerImplementation.getInstance();
	}
	
	@Override
	public UserType createUserType(final String name, final boolean abs, final boolean transaction)
			throws PersistenceException {
		final UserType result = new UserType(IDManager.instance().pullNextUnusedTypeID(name), name, abs, transaction);
		try (final CallableStatement call =
				this.database.getConnection().prepareCall(
						"begin " + this.database.getSchemaName() + ".classfacade.createUserType(?,?,?,?); end;")) {
			call.setLong(1, result.getId());
			call.setString(2, result.getName());
			call.setInt(IntegerConstants.THREE, result.isAbs() ? 1 : 0);
			call.setInt(IntegerConstants.FOUR, result.isTrans() ? 1 : 0);
			call.execute();
		} catch (final SQLException e) {
			throw new OtherSQLException(e);
		}
		this.typeManager.saveType(result);
		return result;
	}
	
	@Override
	public UnidirectionalAssociation createUnidirectionalAssociation(final String name,
			final boolean essential,
			final boolean unique,
			final UserType owner,
			final Type target) throws PersistenceException {
		final UnidirectionalAssociation result =
				new UnidirectionalAssociation(IDManager.instance().pullNextUnusedAssociationID(name), name, owner,
						target, essential, unique);
		try (final CallableStatement call =
				this.database.getConnection().prepareCall(
						"begin " + this.database.getSchemaName()
								+ ".classfacade.createUnidirAssociation(?,?,?,?,?,?); end;")) {
			call.setLong(1, result.getId());
			call.setString(2, result.getName());
			call.setLong(IntegerConstants.THREE, result.getOwner().getId());
			call.setLong(IntegerConstants.FOUR, result.getTarget().getId());
			call.setInt(IntegerConstants.FIVE, result.isEssential() ? 1 : 0);
			call.setInt(IntegerConstants.SIX, result.isUnique() ? 1 : 0);
			call.execute();
		} catch (final SQLException e) {
			throw new OtherSQLException(e);
		}
		this.typeManager.saveAssociation(result);
		return result;
	}
	
	@Override
	public MapAssociation createMapAssociation(final String name,
			final boolean essential,
			final UserType owner,
			final Type target,
			final Type keyType) throws PersistenceException {
		final MapAssociation result =
				new MapAssociation(IDManager.instance().pullNextUnusedAssociationID(name), name, owner, target,
						keyType, essential);
		try (final CallableStatement call =
				this.database.getConnection().prepareCall(
						"begin " + this.database.getSchemaName()
								+ ".classfacade.createMapAssociation(?,?,?,?,?,?,?); end;")) {
			call.setLong(1, result.getId());
			call.setString(2, result.getName());
			call.setLong(IntegerConstants.THREE, result.getOwner().getId());
			call.setLong(IntegerConstants.FOUR, result.getTarget().getId());
			call.setInt(IntegerConstants.FIVE, result.isEssential() ? 1 : 0);
			call.setInt(IntegerConstants.SIX, 1);
			call.setLong(IntegerConstants.SEVEN, result.getKeyType().getId());
			call.execute();
		} catch (final SQLException e) {
			throw new OtherSQLException(e);
		}
		this.typeManager.saveAssociation(result);
		return result;
	}
	
	@Override
	public void createSpecializationBetween(final UserType ancestor, final Type descendant) throws PersistenceException {
		try (final CallableStatement call =
				this.database.getConnection().prepareCall(
						"begin " + this.database.getSchemaName() + ".classfacade.createSpecialization(?,?); end;")) {
			call.setLong(1, ancestor.getId());
			call.setLong(2, descendant.getId());
			call.execute();
		} catch (final SQLException e) {
			throw new OtherSQLException(e);
		}
	}
	
	@Override
	public void initialize() throws PersistenceException {
		try (final CallableStatement call =
				this.database.getConnection().prepareCall(
						"begin " + this.database.getSchemaName() + ".classfacade.initialize; end;")) {
			call.execute();
		} catch (final SQLException e) {
			throw new OtherSQLException(e);
		}
		
		try (final CallableStatement call =
				this.database.getConnection().prepareCall(
						"begin ? :=" + this.database.getSchemaName() + ".classfacade.getMaxIdFromType; end;")) {
			call.registerOutParameter(1, OracleTypes.NUMBER);
			call.execute();
			if (IDManager.getMaxBaseTypeID() != call.getLong(1)) {
				throw new InitializingDatabaseContractViolationException();
			}
		} catch (final SQLException e) {
			throw new OtherSQLException(e);
		}
		
		try (final CallableStatement call =
				this.database.getConnection().prepareCall(
						"begin ? :=" + this.database.getSchemaName() + ".classfacade.getMaxIdFromAssociation; end;")) {
			call.registerOutParameter(1, OracleTypes.NUMBER);
			call.execute();
			if (IDManager.getMaxAssociationContractID() != call.getLong(1)) {
				throw new InitializingDatabaseContractViolationException();
			}
		} catch (final SQLException e) {
			throw new OtherSQLException(e);
		}
		
		this.initializeBaseTypes();
		this.finalizeSpecialization();
		this.initialized = true;
	}
	
	@Override
	public boolean isSuperClassTo(final Type ancestor, final Type descendant) throws PersistenceException {
		try (final CallableStatement call =
				this.database.getConnection().prepareCall(
						"begin ?:=" + this.database.getSchemaName() + ".classfacade.isSuperclassTo(?,?); end;")) {
			call.registerOutParameter(1, OracleTypes.NUMBER);
			call.setLong(2, ancestor.getId());
			call.setLong(IntegerConstants.THREE, descendant.getId());
			call.execute();
			return call.getInt(1) == 1 ? true : false;
		} catch (final SQLException e) {
			e.printStackTrace();
			throw new OtherSQLException(e);
		}
	}
	
	@Override
	public void clear() throws PersistenceException {
		try (final CallableStatement call =
				this.database.getConnection().prepareCall(
						"begin " + this.database.getSchemaName() + ".classfacade.clearAll; end;")) {
			call.execute();
		} catch (final SQLException e) {
			throw new OtherSQLException(e);
		}
	}
	
	@Override
	public void finalizeSpecialization() throws PersistenceException {
		try (final CallableStatement call =
				this.database.getConnection().prepareCall(
						"begin " + this.database.getSchemaName() + ".classfacade.finalizeSpecialization; end;")) {
			call.execute();
		} catch (final SQLException e) {
			if (e.getErrorCode() == SpecializationCycleDetected.ERRORCODE) {
				throw new SpecializationCycleDetected(e);
			}
			e.printStackTrace();
			throw new OtherSQLException(e);
		}
	}
	
	@Override
	public void renameType(final Long typeId, final String newName) throws PersistenceException {
		try (final CallableStatement call =
				this.database.getConnection().prepareCall(
						"begin " + this.database.getSchemaName() + ".classfacade.renameType(?,?); end;")) {
			call.setLong(1, typeId);
			call.setString(2, newName);
			call.execute();
		} catch (final SQLException e) {
			e.printStackTrace();
			throw new OtherSQLException(e);
		}
	}
	
	@Override
	public void renameAssociation(final Long associationId, final String newName) throws PersistenceException {
		try (final CallableStatement call =
				this.database.getConnection().prepareCall(
						"begin " + this.database.getSchemaName() + ".classfacade.renameAssociation(?,?); end;")) {
			call.setLong(1, associationId);
			call.setString(2, newName);
			call.execute();
		} catch (final SQLException e) {
			e.printStackTrace();
			throw new OtherSQLException(e);
		}
	}
	
	@Override
	public void deleteAssociation(final Long assoId) throws PersistenceException {
		this.typeManager.deleteAssociation(assoId);
		try (final CallableStatement call =
				this.database.getConnection().prepareCall(
						"begin " + this.database.getSchemaName() + ".classfacade.deleteAssociation(?); end;")) {
			call.setLong(1, assoId);
			call.execute();
		} catch (final SQLException e) {
			e.printStackTrace();
			throw new OtherSQLException(e);
		}
	}
	
	@Override
	public void updateLinksToNewAssociation(final Long associationId, final Collection<Long> newAssociationIds)
			throws PersistenceException {
		try (final CallableStatement call =
				this.database.getConnection().prepareCall(
						"begin " + this.database.getSchemaName() + ".classfacade.pushDownLinks(?,?); end;")) {
			call.setLong(1, associationId);
			final ArrayDescriptor des =
					ArrayDescriptor.createDescriptor(
							this.database.getSchemaName().toUpperCase() + ".ARRAY_INT",
							this.database.getConnection());
			final Long[] newAssos = newAssociationIds.toArray(new Long[newAssociationIds.size()]);
			final ARRAY arrayNewAssociation = new ARRAY(des, this.database.getConnection(), newAssos);
			call.setArray(2, arrayNewAssociation);
			call.execute();
		} catch (final SQLException e) {
			e.printStackTrace();
			throw new OtherSQLException(e);
		}
	}
	
	@Override
	public void deleteUserType(final Long typeId) throws PersistenceException {
		this.typeManager.deleteType(typeId);
		try (final CallableStatement call =
				this.database.getConnection().prepareCall(
						"begin " + this.database.getSchemaName() + ".classfacade.deleteUserTypeAndSpec(?); end;")) {
			call.setLong(1, typeId);
			call.execute();
		} catch (final SQLException e) {
			e.printStackTrace();
			throw new OtherSQLException(e);
		}
	}
	
	@Override
	public void moveLinksAndCreateObjects(final List<Long> oldAssoIds,
			final Association newAsso,
			final UserType newType,
			final List<Long> newAssoIds) throws PersistenceException {
		try (final CallableStatement call =
				this.database.getConnection().prepareCall(
						"begin " + this.database.getSchemaName()
								+ ".classfacade.moveLinksAndCreateObjects(?,?,?,?); end;")) {
			// TODO Implement database function!
			final ArrayDescriptor des =
					ArrayDescriptor.createDescriptor(
							this.database.getSchemaName().toUpperCase() + ".ARRAY_INT",
							this.database.getConnection());
			final Long[] oldAssosA = oldAssoIds.toArray(new Long[oldAssoIds.size()]);
			final Long[] newAssosA = newAssoIds.toArray(new Long[newAssoIds.size()]);
			final ARRAY arrayNewAssociations = new ARRAY(des, this.database.getConnection(), newAssosA);
			final ARRAY arrayOldAssociations = new ARRAY(des, this.database.getConnection(), oldAssosA);
			
			call.setArray(1, arrayOldAssociations);
			call.setLong(2, newAsso.getId());
			call.setLong(IntegerConstants.THREE, newType.getId());
			call.setArray(IntegerConstants.FOUR, arrayNewAssociations);
			call.execute();
		} catch (final SQLException e) {
			e.printStackTrace();
			throw new OtherSQLException(e);
		}
	}
	
	@Override
	public TypeManager getTypeManager() throws ClassFacadeUninitializedException {
		if (!this.hasBeenInitialized()) {
			throw new ClassFacadeUninitializedException();
		}
		return this.typeManager;
	}
	
	private void initializeBaseTypes() {
		this.typeManager.saveType(StringType.getInstance());
		this.typeManager.saveType(IntegerType.getInstance());
	}
	
	/**
	 * A private method to check if an Object exists in the types Collection with the type information (id, name) of the
	 * String Type from the contract and removes this object from the types Collection, if it exists.
	 * 
	 * @param types
	 *            A Collection of Types.
	 * @return Provides true if an Object with the string type information exists.
	 */
	private boolean checkIfStringIsCorrect(final Collection<Type> types) {
		final Iterator<Type> i = types.iterator();
		while (i.hasNext()) {
			final Type current = i.next();
			if (current.getId() == 1 && current.getName().equals(StringType.STRING_NAME)) {
				i.remove();
				return true;
			}
		}
		return false;
	}
	
	/**
	 * A private method to check if an Object exists in the types Collection with the type information (id, name) of the
	 * Integer Type from the contract and removes this object from the types Collection, if it exists.
	 * 
	 * @param types
	 *            A Collection of Types.
	 * @return Provides true if an Object with the integer type information exists.
	 */
	private boolean checkIfIntegerIsCorrect(final Collection<Type> types) {
		final Iterator<Type> i = types.iterator();
		while (i.hasNext()) {
			final Type current = i.next();
			if (current.getId() == 2 && current.getName().equals(IntegerType.INTEGER_NAME)) {
				i.remove();
				return true;
			}
		}
		return false;
	}
	
	@Override
	public boolean hasBeenInitialized() {
		return this.initialized;
	}
	
	@Override
	public void initializeForRuntime() throws PersistenceException {
		this.typeManager = TypeManagerImplementation.getInstance();
		final Collection<Type> types = new Vector<>();
		try (final CallableStatement call =
				this.database.getConnection().prepareCall(
						"begin ?:= " + this.database.getSchemaName() + ".classfacade.getAllTypes; end;")) {
			call.registerOutParameter(1, OracleTypes.CURSOR);
			call.execute();
			
			try (final ResultSet result = ((OracleCallableStatement) call).getCursor(1)) {
				while (result.next()) {
					types.add(new UserType(result.getLong(1), result.getString(2), result
							.getInt(IntegerConstants.THREE) == 1 ? true : false,
							result.getInt(IntegerConstants.FOUR) == 1 ? true : false));
				}
			}
		} catch (final SQLException e) {
			throw new OtherSQLException(e);
		}
		
		if (!this.checkIfStringIsCorrect(types)) {
			throw new InitializingDatabaseContractViolationException();
		}
		if (!this.checkIfIntegerIsCorrect(types)) {
			throw new InitializingDatabaseContractViolationException();
		}
		this.initializeBaseTypes();
		for (final Type t : types) {
			this.typeManager.saveType(t);
		}
		
		try (final CallableStatement call =
				this.database.getConnection().prepareCall(
						"begin ?:= " + this.database.getSchemaName() + ".classfacade.getAllUnidirAssociations; end;")) {
			call.registerOutParameter(1, OracleTypes.CURSOR);
			call.execute();
			
			final Collection<UnidirectionalAssociation> unidirectionalAssociations = new Vector<>();
			try (final ResultSet result = ((OracleCallableStatement) call).getCursor(1)) {
				while (result.next()) {
					unidirectionalAssociations.add(new UnidirectionalAssociation(result.getLong(1),
							result.getString(2), (UserType) this.typeManager.getTypeForId(result
									.getLong(IntegerConstants.THREE)), this.typeManager.getTypeForId(result
									.getLong(IntegerConstants.FOUR)), result.getInt(IntegerConstants.FIVE) == 1 ? true
									: false, result.getInt(IntegerConstants.SIX) == 1 ? true : false));
				}
			}
			final Iterator<UnidirectionalAssociation> i = unidirectionalAssociations.iterator();
			while (i.hasNext()) {
				final UnidirectionalAssociation current = i.next();
				this.typeManager.saveAssociation(current);
			}
		} catch (final SQLException e) {
			throw new OtherSQLException(e);
		}
		
		try (final CallableStatement call =
				this.database.getConnection().prepareCall(
						"begin ?:= " + this.database.getSchemaName() + ".classfacade.getAllMapAssociations; end;")) {
			call.registerOutParameter(1, OracleTypes.CURSOR);
			call.execute();
			
			final Collection<MapAssociation> mapAssociations = new Vector<>();
			try (final ResultSet result = ((OracleCallableStatement) call).getCursor(1)) {
				while (result.next()) {
					mapAssociations.add(new MapAssociation(result.getLong(1), result.getString(2),
							(UserType) this.typeManager.getTypeForId(result.getLong(IntegerConstants.THREE)),
							this.typeManager.getTypeForId(result.getLong(IntegerConstants.FOUR)), this.typeManager
									.getTypeForId(result.getLong(IntegerConstants.FIVE)), result
									.getInt(IntegerConstants.SIX) == 1 ? true : false));
				}
			}
			final Iterator<MapAssociation> i = mapAssociations.iterator();
			while (i.hasNext()) {
				final MapAssociation current = i.next();
				this.typeManager.saveAssociation(current);
			}
		} catch (final SQLException e) {
			throw new OtherSQLException(e);
		}
		
		this.initialized = true;
		// TODO set the lastSetTypeId and the lastSetAssociationId correctly
	}
	
}
