package de.fhdw.wtf.persistence.facade;

import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.sql.CallableStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.HashSet;
import java.util.Set;

import oracle.jdbc.OracleTypes;
import oracle.jdbc.internal.OracleCallableStatement;
import de.fhdw.wtf.persistence.exception.MandantNotLoggedInException;
import de.fhdw.wtf.persistence.exception.NameAlreadyExistsException;
import de.fhdw.wtf.persistence.exception.ObjectNotFoundException;
import de.fhdw.wtf.persistence.exception.OtherSQLException;
import de.fhdw.wtf.persistence.exception.PersistenceException;
import de.fhdw.wtf.persistence.exception.WrongPasswordException;
import de.fhdw.wtf.persistence.meta.Mandant;
import de.fhdw.wtf.persistence.meta.Role;
import de.fhdw.wtf.persistence.meta.User;
import de.fhdw.wtf.persistence.utils.IntegerConstants;

/**
 * Class for the authorization at the Oracle Database.
 * 
 */
public class OracleAuthorizationFacadeImplementation implements AuthorizationFacade {
	
	private boolean mandantLoggedIn;
	
	private Mandant currentMandant;
	
	private final String salt;
	
	private final OracleDatabaseManager database;
	
	/**
	 * TODO.
	 * 
	 * @param salt
	 * @param database
	 *            is the Database all future Operations will use for the authorization.
	 */
	public OracleAuthorizationFacadeImplementation(final String salt, final OracleDatabaseManager database) {
		super();
		this.salt = salt;
		this.database = database;
		this.currentMandant = null;
		this.mandantLoggedIn = false;
	}
	
	@Override
	public void logIn(final Mandant mandant) throws PersistenceException {
		this.currentMandant = mandant;
		if (!this.isPasswordCorrect(this.currentMandant.getId(), "root", this.currentMandant.getRootPassword())) {
			throw new WrongPasswordException(mandant.getId(), "root");
		}
		try (final CallableStatement call =
				this.database.getConnection().prepareCall("begin ? := accountfacade.getAllUsersForMandant(?); end;")) {
			call.registerOutParameter(1, OracleTypes.CURSOR);
			call.setString(2, this.currentMandant.getId());
			call.execute();
			
			try (final ResultSet result = ((OracleCallableStatement) call).getCursor(1)) {
				while (result.next()) {
					final String name = result.getString(3);
					if (!name.equals("root")) {
						final User entry = new User(result.getLong(1), name, result.getString(2));
						this.currentMandant.addUser(entry);
					}
				}
			}
			this.mandantLoggedIn = true;
		} catch (final SQLException e) {
			throw new OtherSQLException(e);
		}
	}
	
	@Override
	public boolean isPasswordCorrect(final String mandant, final String username, final String password)
			throws PersistenceException {
		try {
			final MessageDigest message = MessageDigest.getInstance("MD5");
			message.update((password + this.salt).getBytes());
			final String hashedPW = toHex(message.digest());
			
			try (final CallableStatement call =
					this.database.getConnection().prepareCall("begin ? := accountfacade.authorize(?,?,?); end;")) {
				call.registerOutParameter(1, OracleTypes.NUMBER);
				call.setString(2, mandant);
				call.setString(IntegerConstants.THREE, username);
				call.setString(IntegerConstants.FOUR, hashedPW);
				call.execute();
				return call.getInt(1) == 1;
			} catch (final SQLException e) {
				throw new OtherSQLException(e);
			}
		} catch (final NoSuchAlgorithmException e) {
			e.printStackTrace();
			return false;
		}
	}
	
	private void checkMandantLogIn() throws PersistenceException {
		if (!this.mandantLoggedIn) {
			throw new MandantNotLoggedInException();
		}
	}
	
	@Override
	public User createUser(final String username, final String password) throws PersistenceException {
		this.checkMandantLogIn();
		final long id = this.createUser(this.currentMandant.getId(), username, password);
		final User result = new User(id, username, this.currentMandant.getId());
		this.currentMandant.addUser(result);
		return result;
	}
	
	@Override
	public void grantRoleToUser(final User user, final Role role) throws PersistenceException {
		this.checkMandantLogIn();
		try (final CallableStatement call =
				this.database.getConnection().prepareCall("begin accountfacade.addRoleToUser(?,?); end;")) {
			call.setLong(1, user.getId());
			call.setLong(2, role.getId());
			call.execute();
		} catch (final SQLException e) {
			throw new OtherSQLException(e);
		}
		
	}
	
	@Override
	public Role createRole(final String name) throws PersistenceException {
		try (final CallableStatement call =
				this.database.getConnection().prepareCall("begin ? := accountfacade.createRole(?); end;")) {
			call.registerOutParameter(1, OracleTypes.NUMBER);
			call.setString(2, name);
			call.execute();
			return new Role(call.getLong(1), name);
		} catch (final SQLException e) {
			if (e.getErrorCode() == NameAlreadyExistsException.ERRORCODE) {
				throw new NameAlreadyExistsException(e);
			}
			e.printStackTrace();
			throw new OtherSQLException(e);
		}
	}
	
	@Override
	public Set<Role> getRolesFrom(final User user) throws PersistenceException {
		try (final CallableStatement call =
				this.database.getConnection().prepareCall("begin ? := accountfacade.getRoles(?,?); end;")) {
			call.registerOutParameter(1, OracleTypes.CURSOR);
			call.setString(2, user.getMandantId());
			call.setString(IntegerConstants.THREE, user.getName());
			call.execute();
			try (final ResultSet resultSet = ((OracleCallableStatement) call).getCursor(1)) {
				final Set<Role> result = new HashSet<>();
				while (resultSet.next()) {
					final Role entry = new Role(resultSet.getLong(1), resultSet.getString(2));
					result.add(entry);
				}
				return result;
			}
		} catch (final SQLException e) {
			throw new OtherSQLException(e);
		}
	}
	
	private long createUser(final String mandant, final String username, final String password)
			throws PersistenceException {
		try {
			final MessageDigest message = MessageDigest.getInstance("MD5");
			message.update((password + this.salt).getBytes());
			final String hashedPassword = toHex(message.digest());
			
			try (final CallableStatement call =
					this.database.getConnection().prepareCall("begin ? := accountfacade.createUser(?,?,?); end;")) {
				call.registerOutParameter(1, OracleTypes.NUMBER);
				call.setString(2, mandant);
				call.setString(IntegerConstants.THREE, username);
				call.setString(IntegerConstants.FOUR, hashedPassword);
				call.execute();
				return call.getLong(1);
			} catch (final SQLException e) {
				if (e.getErrorCode() == NameAlreadyExistsException.ERRORCODE) {
					throw new NameAlreadyExistsException(e);
				}
				throw new OtherSQLException(e);
			}
		} catch (final NoSuchAlgorithmException e) {
			e.printStackTrace();
			throw new Error(e);
		}
	}
	
	@Override
	public Mandant createMandant(final String id, final String rootPassword) throws PersistenceException {
		this.createUser(id, "root", rootPassword);
		return new Mandant(id, rootPassword);
	}
	
	@Override
	public Set<User> getUsers() throws PersistenceException {
		this.checkMandantLogIn();
		return this.currentMandant.getUsers();
	}
	
	@Override
	public User getUser(final String mandant, final String name) throws PersistenceException {
		try (final CallableStatement call =
				this.database.getConnection().prepareCall("begin ? := accountfacade.getUser(?,?); end;")) {
			call.registerOutParameter(1, OracleTypes.CURSOR);
			call.setString(2, mandant);
			call.setString(IntegerConstants.THREE, name);
			call.execute();
			
			try (final ResultSet resultSet = ((OracleCallableStatement) call).getCursor(1)) {
				if (resultSet.next()) {
					return new User(resultSet.getLong(1), resultSet.getString(IntegerConstants.THREE),
							resultSet.getString(2));
				} else {
					// TODO: create UserNotFoundException
					throw new ObjectNotFoundException();
				}
			}
		} catch (final SQLException e) {
			e.printStackTrace();
			throw new OtherSQLException(e);
		}
	}
	
	@Override
	public Set<Role> listAllRoles() throws PersistenceException {
		final Set<Role> result = new HashSet<>();
		try (final CallableStatement call =
				this.database.getConnection().prepareCall("begin ?:= accountfacade.getRoles; end;")) {
			call.registerOutParameter(1, OracleTypes.CURSOR);
			call.execute();
			
			try (final ResultSet resultSet = ((OracleCallableStatement) call).getCursor(1)) {
				while (resultSet.next()) {
					result.add(new Role(resultSet.getLong(1), resultSet.getString(2)));
				}
			}
		} catch (final SQLException e) {
			throw new OtherSQLException(e);
		}
		
		return result;
	}
	
	private static final char[] hexArray = "0123456789ABCDEF".toCharArray();
	
	private static String toHex(final byte[] bytes) {
		final char[] hexChars = new char[bytes.length * 2];
		for (int i = 0; i < bytes.length; i++) {
			final int v = bytes[i] & 0xFF;
			hexChars[i * 2] = hexArray[v >>> IntegerConstants.FOUR];
			hexChars[i * 2 + 1] = hexArray[v & 0x0F];
		}
		return new String(hexChars);
	}
	
}
