package de.fhdw.wtf.persistence.facade;

import java.math.BigInteger;
import java.util.Collection;
import java.util.Date;

import de.fhdw.wtf.persistence.exception.PersistenceException;
import de.fhdw.wtf.persistence.meta.Association;
import de.fhdw.wtf.persistence.meta.Link;
import de.fhdw.wtf.persistence.meta.MapAssociation;
import de.fhdw.wtf.persistence.meta.MapLink;
import de.fhdw.wtf.persistence.meta.Object;
import de.fhdw.wtf.persistence.meta.Transaction;
import de.fhdw.wtf.persistence.meta.UnidirectionalAssociation;
import de.fhdw.wtf.persistence.meta.UnidirectionalLink;
import de.fhdw.wtf.persistence.meta.UserObject;
import de.fhdw.wtf.persistence.meta.UserType;
import de.fhdw.wtf.persistence.utils.Tuple;

/**
 * An Interface to represent a facade towards the database to manipulate the Instance Layer of the database.
 * 
 */
public interface ObjectFacade {
	
	/**
	 * Wildcard character in use for Kleene closure
	 */
	char REGEX_CHAR_KLEENE_STAR = '%';
	
	// ---------- Date:Find ----------
	
	/**
	 * Provides a Collection of {@link de.fhdw.wtf.persistence.meta.UserObject User Objects} which are owners of the
	 * {@link de.fhdw.wtf.persistence.meta.Association Association} {@code association}, which points to a String value
	 * which is like {@code string} and the link between the user Object and the String has been published before
	 * {@code date} and if the link has been deleted, the deletion date must be before {@code date}. SQL wildcards in
	 * {@code string} are allowed.
	 * 
	 * @param association
	 *            An UnidirectionalAssociation, which target is a String
	 * @param string
	 *            A given String value
	 * @param date
	 *            The moment in time the object graph is queried.
	 * @return A Set of User Objects.
	 * @throws PersistenceException
	 *             This exception occurs if there are any problems contacting the database.
	 */
	Collection<UserObject> find(Association association, String string, Date date) throws PersistenceException;
	
	/**
	 * Provides a Collection of User Objects which are owners of the Association association, which points to a Integer
	 * value which is like s and the link between the user Object and the String has been published before d and if the
	 * link has been deleted, the deletion date must be before d.
	 * 
	 * @param association
	 *            An UnidirectionalAssociation, which target is an Integer
	 * @param integer
	 *            A given Integer value
	 * @param date
	 *            The moment in time the object graph is queried.
	 * @return A Set of User Objects.
	 * @throws PersistenceException
	 *             This exception occurs if there are any problems contacting the database.
	 */
	Collection<UserObject> find(Association association, BigInteger integer, Date date) throws PersistenceException;
	
	// ---------- Date:Get ----------
	
	/**
	 * Provides a collection of Tuples of Links and Objects. The Links are instances of association and their owner is
	 * owner. The Objects are the targets of each link. The published date of the link are before date and possible
	 * deletion dates of the links are after date. So date specifies the moment of time the object graph is queried.
	 * 
	 * @param owner
	 *            An User Object.
	 * @param association
	 *            An association where the owner points to type(o)
	 * @param date
	 *            A moment in time the object graph is queried.
	 * @return A Set of Tuples, where the first component is a link and the second component is an Object.
	 * @throws PersistenceException
	 *             This exception occurs if there are any problems contacting the database.
	 */
	Collection<Tuple<UnidirectionalLink, Object>> get(UserObject owner, UnidirectionalAssociation association, Date date)
			throws PersistenceException;
	
	// ---------- Date:Get:Map ----------
	
	/**
	 * Provides a collection of Objects for a map. The Links are instances of association and their owner is owner. The
	 * Objects are the targets of each link. The published date of the link are before date and possible deletion dates
	 * of the links are after date. So date specifies the moment of time the object graph is queried.
	 * 
	 * @param owner
	 *            An User Object.
	 * @param association
	 *            An association where the owner points to type(o)
	 * @param key
	 *            The map-key
	 * @param date
	 *            A moment in time the object graph is queried.
	 * @return A Set of Tuples, where the first component is a link and the second component is an Object.
	 * @throws PersistenceException
	 *             This exception occurs if there are any problems contacting the database.
	 */
	Collection<Object> get(UserObject owner, MapAssociation association, String key, Date date)
			throws PersistenceException;
	
	/**
	 * Provides a collection of Objects for a map. The Links are instances of association and their owner is owner. The
	 * Objects are the targets of each link. The published date of the link are before date and possible deletion dates
	 * of the links are after date. So date specifies the moment of time the object graph is queried.
	 * 
	 * @param owner
	 *            An User Object.
	 * @param association
	 *            An association where the owner points to type(o)
	 * @param key
	 *            The map-key
	 * @param date
	 *            A moment in time the object graph is queried.
	 * @return A Set of Tuples, where the first component is a link and the second component is an Object.
	 * @throws PersistenceException
	 *             This exception occurs if there are any problems contacting the database.
	 */
	Collection<Object> get(UserObject owner, MapAssociation association, BigInteger key, Date date)
			throws PersistenceException;
	
	/**
	 * Provides a collection of Objects for a map. The Links are instances of association and their owner is owner. The
	 * Objects are the targets of each link. The published date of the link are before date and possible deletion dates
	 * of the links are after date. So date specifies the moment of time the object graph is queried.
	 * 
	 * @param owner
	 *            An User Object.
	 * @param association
	 *            An association where the owner points to type(o)
	 * @param key
	 *            The map-key
	 * @param date
	 *            A moment in time the object graph is queried.
	 * @return A Set of Tuples, where the first component is a link and the second component is an Object.
	 * @throws PersistenceException
	 *             This exception occurs if there are any problems contacting the database.
	 */
	Collection<Object> get(UserObject owner, MapAssociation association, UserObject key, Date date)
			throws PersistenceException;
	
	// ---------- Date:InverseGet ----------
	
	/**
	 * Provides a collection of Tuples of Links and UserObjects. The Links are instances of a and their owner is o. The
	 * Objects are the targets of each link. The published date of the link are before d and possible deletion dates of
	 * the links are after d. So d specifies the moment of time the object graph is queried.
	 * 
	 * @param s
	 *            A String.
	 * @param association
	 *            An association where the owner points to type(o)
	 * @param date
	 *            A moment in time the object graph is queried.
	 * @return A Set of Tuples, where the first component is a UnidirectionalLink and the second component is an user
	 *         object.
	 * @throws PersistenceException
	 *             This exception occurs if there are any problems contacting the database.
	 */
	// Collection<Tuple<UnidirectionalLink, UserObject>>
	// inverseGet(String s, UnidirectionalAssociation association, Date date)
	// throws PersistenceException;
	
	/**
	 * Provides a collection of Tuples of Links and Objects. The Links are instances of a and their owner is o. The
	 * Objects are the targets of each link. The published date of the link are before d and possible deletion dates of
	 * the links are after d. So d specifies the moment of time the object graph is queried.
	 * 
	 * @param i
	 *            An Interger.
	 * @param association
	 *            An association where the owner points to type(o)
	 * @param date
	 *            A moment in time the object graph is queried.
	 * @return A Set of Tuples, where the first component is a UnidirectionalLink and the second component is an user
	 *         object.
	 * @throws PersistenceException
	 *             This exception occurs if there are any problems contacting the database.
	 */
	// Collection<Tuple<UnidirectionalLink, UserObject>>
	// inverseGet(Integer i, UnidirectionalAssociation association, Date date)
	// throws PersistenceException;
	
	/**
	 * Provides a collection of Tuples of Links and Objects. The Links are instances of association and their target is
	 * target. The Objects are the targets of each link. The published date of the link are before date and possible
	 * deletion dates of the links are after date. So date specifies the moment of time the object graph is queried.
	 * 
	 * @param target
	 *            An User Object.
	 * @param association
	 *            An association where the owner points to type(o)
	 * @param date
	 *            A moment in time the object graph is queried.
	 * @return A Set of Tuples, where the first component is a UnidirectionalLink and the second component is an user
	 *         object.
	 * @throws PersistenceException
	 *             This exception occurs if there are any problems contacting the database.
	 */
	Collection<Tuple<UnidirectionalLink, UserObject>> inverseGet(UserObject target,
			UnidirectionalAssociation association,
			Date date) throws PersistenceException;
	
	// ---------- Date:InverseGet:Map ----------
	
	/**
	 * Provides a owner for a map association. The UserObjects are the owner of each link. The published date of the
	 * link are before date and possible deletion dates of the links are after date. So date specifies the moment of
	 * time the object graph is queried.
	 * 
	 * @param target
	 *            An User Object represents the target.
	 * @param association
	 *            An association where the owner points to type(o)
	 * @param date
	 *            A moment in time the object graph is queried.
	 * @return A Set of Tuples, where the first component is the owner and the second is the key.
	 * @throws PersistenceException
	 *             This exception occurs if there are any problems contacting the database.
	 */
	Collection<Tuple<UserObject, Object>> inverseGet(UserObject target, MapAssociation association, Date date)
			throws PersistenceException;
	
	/**
	 * Provides a owner for a map association. The UserObjects are the owner of each link. The published date of the
	 * link are before date and possible deletion dates of the links are after date. So date specifies the moment of
	 * time the object graph is queried.
	 * 
	 * @param target
	 *            An User Object represents the target.
	 * @param association
	 *            An association where the owner points to type(o)
	 * @param date
	 *            A moment in time the object graph is queried.
	 * @return A Set of Tuples, where the first component is the owner and the second is the key.
	 * @throws PersistenceException
	 *             This exception occurs if there are any problems contacting the database.
	 */
	Collection<Tuple<UserObject, Object>> inverseGet(String target, MapAssociation association, Date date)
			throws PersistenceException;
	
	//
	/**
	 * Provides a owner for a map association. The UserObjects are the owner of each link. The published date of the
	 * link are before date and possible deletion dates of the links are after date. So date specifies the moment of
	 * time the object graph is queried.
	 * 
	 * @param target
	 *            An User Object represents the target.
	 * @param association
	 *            An association where the owner points to type(o)
	 * @param date
	 *            A moment in time the object graph is queried.
	 * @return A Set of Tuples, where the first component is the owner and the second is the key.
	 * @throws PersistenceException
	 *             This exception occurs if there are any problems contacting the database.
	 */
	Collection<Tuple<UserObject, Object>> inverseGet(BigInteger target, MapAssociation association, Date date)
			throws PersistenceException;
	
	// ---------- Trans:Find ----------
	
	/**
	 * Provides a Set of User Objects which have an association to a String Value, which value is like string and which
	 * are in the scope of the transaction transaction. User Objects which are in the scope of transaction are created
	 * by transaction or currently published and not not yet deleted by transaction.
	 * 
	 * @param association
	 *            An UnidirectionalAssociation towards a String, where the owner is a supertype to the type of the User
	 *            Objects.
	 * @param string
	 *            A given String value. SQL Wildcards are allowed.
	 * @param transaction
	 *            A currently open Transaction.
	 * @return A Set of User Objects.
	 * @throws PersistenceException
	 *             This exception occurs if there are any problems contacting the database.
	 */
	Collection<UserObject> find(Association association, String string, Transaction transaction)
			throws PersistenceException;
	
	/**
	 * Provides a Set of User Objects which have an association to an Integer Value, which value is like integer and
	 * which are in the scope of the transaction. User Objects which are in the scope of transaction are created by
	 * transaction or currently published and not not yet deleted by transaction.
	 * 
	 * @param association
	 *            An UnidirectionalAssociation towards an Integer, where the owner is a supertype to the type of the
	 *            User Objects.
	 * @param integer
	 *            A given Integer Value.
	 * @param transaction
	 *            A currently open Transaction.
	 * @return A Set of User Objects.
	 * @throws PersistenceException
	 *             This exception occurs if there are any problems contacting the database.
	 */
	Collection<UserObject> find(Association association, BigInteger integer, Transaction transaction)
			throws PersistenceException;
	
	// ---------- Trans:Get ----------
	
	/**
	 * Provides a collection of Tuples of Links and Objects. The Links are instances of the UnidirectionalAssociation
	 * association. And the Objects are the owner of the Links. type(owner) has to be is subtype of the owner of the
	 * UnidirectionalAssociation association. The Links and Objects are in the scope of the Transaction transaction.
	 * This means, that they are either created by transaction or currently published and not yet deleted.
	 * 
	 * @param owner
	 *            A User Objects, which type has an association a.
	 * @param association
	 *            An UnidirectionalAssociation with type(o) = owner
	 * @param transaction
	 *            A currently open Transaction.
	 * @return A Set of Tuples, where the first component is a link and the second component is an Object.
	 * @throws PersistenceException
	 *             This exception occurs if there are any problems contacting the database.
	 */
	Collection<Tuple<UnidirectionalLink, Object>> get(UserObject owner,
			UnidirectionalAssociation association,
			Transaction transaction) throws PersistenceException;
	
	// ---------- Trans:Get:Map ----------
	
	/**
	 * Provides the target from a map link. The Links are instances of the MapAssociation association. And the Objects
	 * are the owner of the Links. type(owner) has to be is subtype of the owner of the MapAssociation association. The
	 * Links and Objects are in the scope of the Transaction transaction. This means, that they are either created by
	 * transaction or currently published and not yet deleted.
	 * 
	 * @param owner
	 *            A User Objects, which type has an association a.
	 * @param association
	 *            An UnidirectionalAssociation with type(o) = owner
	 * @param key
	 *            The map-key
	 * @param transaction
	 *            A currently open Transaction.
	 * @return A Set of Tuples, where the first component is a link and the second component is an Object.
	 * @throws PersistenceException
	 *             This exception occurs if there are any problems contacting the database.
	 */
	Collection<Tuple<MapLink, Object>> get(UserObject owner,
			MapAssociation association,
			String key,
			Transaction transaction) throws PersistenceException;
	
	/**
	 * Provides the target from a map link. The Links are instances of the MapAssociation association. And the Objects
	 * are the owner of the Links. type(owner) has to be is subtype of the owner of the MapAssociation association. The
	 * Links and Objects are in the scope of the Transaction transaction. This means, that they are either created by
	 * transaction or currently published and not yet deleted.
	 * 
	 * @param owner
	 *            A User Objects, which type has an association a.
	 * @param association
	 *            An UnidirectionalAssociation with type(o) = owner
	 * @param key
	 *            The map-key
	 * @param transaction
	 *            A currently open Transaction.
	 * @return A Set of Tuples, where the first component is a link and the second component is an Object.
	 * @throws PersistenceException
	 *             This exception occurs if there are any problems contacting the database.
	 */
	Collection<Tuple<MapLink, Object>> get(UserObject owner,
			MapAssociation association,
			BigInteger key,
			Transaction transaction) throws PersistenceException;
	
	/**
	 * Provides the target from a map link. The Links are instances of the MapAssociation association. And the Objects
	 * are the owner of the Links. type(owner) has to be is subtype of the owner of the MapAssociation association. The
	 * Links and Objects are in the scope of the Transaction transaction. This means, that they are either created by
	 * transaction or currently published and not yet deleted.
	 * 
	 * @param owner
	 *            A User Objects, which type has an association a.
	 * @param association
	 *            An UnidirectionalAssociation with type(o) = owner
	 * @param key
	 *            The map-key
	 * @param transaction
	 *            A currently open Transaction.
	 * @return A Set of Tuples, where the first component is a link and the second component is an Object.
	 * @throws PersistenceException
	 *             This exception occurs if there are any problems contacting the database.
	 */
	Collection<Tuple<MapLink, Object>> get(UserObject owner,
			MapAssociation association,
			UserObject key,
			Transaction transaction) throws PersistenceException;
	
	// ---------- Trans:InverseGet ----------
	
	/**
	 * Provides a collection of Tuples of Links and User Objects. The Links are instances of the
	 * UnidirectionalAssociation association. And the User Objects are the targets of the Links. type(target) has to be
	 * is subtype of the target of the UnidirectionalAssociation association. The Links and Objects are in the scope of
	 * the Transaction transaction. This means, that they are either created by transaction or currently published and
	 * not yet deleted.
	 * 
	 * @param target
	 *            A User Object, which type is target of an association a
	 * @param association
	 *            An UnidirectionalAssociation with type(o) = target
	 * @param transaction
	 *            A currently open Transaction.
	 * @return A Set of Tuples, where the first component is a UnidirectionalLink and the second component is an user
	 *         object.
	 * @throws PersistenceException
	 *             This exception occurs if there are any problems contacting the database.
	 */
	Collection<Tuple<UnidirectionalLink, UserObject>> inverseGet(UserObject target,
			UnidirectionalAssociation association,
			Transaction transaction) throws PersistenceException;
	
	/**
	 * Provides a collection of Tuples of Links and User Objects. The Links are instances of the
	 * UnidirectionalAssociation a. And the User Objects are the owners of the Links. type(o) has to be is subtype of
	 * the target of the UnidirectionalAssociation a. The Links and Objects are in the scope of the Transaction t. This
	 * means, that they are either created by t or currently published and not yet deleted.
	 * 
	 * @param o
	 *            A User Object, which type is target of an association a
	 * @param association
	 *            An UnidirectionalAssociation with type(o) = target
	 * @param transaction
	 *            A currently open Transaction.
	 * @return A Set of Tuples, where the first component is a UnidirectionalLink and the second component is an user
	 *         object.
	 * @throws PersistenceException
	 *             This exception occurs if there are any problems contacting the database.
	 */
	// Collection<Tuple<UnidirectionalLink, UserObject>>
	// inverseGet(String s, UnidirectionalAssociation association, Transaction
	// transaction) throws PersistenceException;
	
	/**
	 * Provides a collection of Tuples of Links and User Objects. The Links are instances of the
	 * UnidirectionalAssociation a. And the User Objects are the owners of the Links. type(o) has to be is subtype of
	 * the target of the UnidirectionalAssociation a. The Links and Objects are in the scope of the Transaction t. This
	 * means, that they are either created by t or currently published and not yet deleted.
	 * 
	 * @param o
	 *            A User Object, which type is target of an association a
	 * @param association
	 *            An UnidirectionalAssociation with type(o) = target
	 * @param transaction
	 *            A currently open Transaction.
	 * @return A Set of Tuples, where the first component is a UnidirectionalLink and the second component is an user
	 *         object.
	 * @throws PersistenceException
	 *             This exception occurs if there are any problems contacting the database.
	 */
	// Collection<Tuple<UnidirectionalLink, UserObject>>
	// inverseGet(Integer i, UnidirectionalAssociation association, Transaction
	// transaction) throws PersistenceException;
	
	// ---------- Trans:InverseGet:Map ----------
	
	/**
	 * Provides a collection of Tuples of the Owner and Key for a map association. The User Objects are the owners.
	 * type(target) has to be is subtype of the target of the MapAssociation association. The Keys and Owner are in the
	 * scope of the Transaction transaction. This means, that they are either created by transaction or currently
	 * published and not yet deleted.
	 * 
	 * @param target
	 *            A User Object, which type is target of an association a
	 * @param association
	 *            An UnidirectionalAssociation with type(o) = target
	 * @param transaction
	 *            A currently open Transaction.
	 * @return A Set of Tuples, where the first component is a UnidirectionalLink and the second component is an user
	 *         object.
	 * @throws PersistenceException
	 *             This exception occurs if there are any problems contacting the database.
	 */
	Collection<Tuple<UserObject, Object>> inverseGet(UserObject target,
			MapAssociation association,
			Transaction transaction) throws PersistenceException;
	
	/**
	 * Provides a collection of Tuples of the Owner and Key for a map association. The User Objects are the owners.
	 * type(target) has to be is subtype of the target of the MapAssociation association. The Keys and Owner are in the
	 * scope of the Transaction transaction. This means, that they are either created by transaction or currently
	 * published and not yet deleted.
	 * 
	 * @param target
	 *            A User Object, which type is target of an association a
	 * @param association
	 *            An UnidirectionalAssociation with type(o) = target
	 * @param transaction
	 *            A currently open Transaction.
	 * @return A Set of Tuples, where the first component is a UnidirectionalLink and the second component is an user
	 *         object.
	 * @throws PersistenceException
	 *             This exception occurs if there are any problems contacting the database.
	 */
	Collection<Tuple<UserObject, Object>> inverseGet(String target, MapAssociation association, Transaction transaction)
			throws PersistenceException;
	
	/**
	 * Provides a collection of Tuples of the Owner and Key for a map association. The User Objects are the owners.
	 * type(target) has to be is subtype of the target of the MapAssociation association. The Keys and Owner are in the
	 * scope of the Transaction transaction. This means, that they are either created by transaction or currently
	 * published and not yet deleted.
	 * 
	 * @param target
	 *            A User Object, which type is target of an association a
	 * @param association
	 *            An UnidirectionalAssociation with type(o) = target
	 * @param transaction
	 *            A currently open Transaction.
	 * @return A Set of Tuples, where the first component is a UnidirectionalLink and the second component is an user
	 *         object.
	 * @throws PersistenceException
	 *             This exception occurs if there are any problems contacting the database.
	 */
	Collection<Tuple<UserObject, Object>> inverseGet(BigInteger target,
			MapAssociation association,
			Transaction transaction) throws PersistenceException;
	
	// ---------- Other ----------
	
	/**
	 * Provides true if before and after are in conflict, e.g. accessing Objects and Links in a modifying manner, when
	 * before is committed before after.
	 * 
	 * @param before
	 *            A currently open transaction.
	 * @param after
	 *            A currently open transaction.
	 * @return Provides true if the two transactions are in conflict, when before commits before after.
	 * @throws PersistenceException
	 *             This exception occurs if there are any problems contacting the database.
	 */
	boolean isInConflict(Transaction before, Transaction after) throws PersistenceException;
	
	/**
	 * Checks if a given Transaction is a transaction which is not yet committed or rolled back if t is an adhoc
	 * transaction.
	 * 
	 * @param transaction
	 *            A possible open Transaction.
	 * @return Provides true if t is an open transaction.
	 * @throws PersistenceException
	 *             This exception occurs if there are any problems contacting the database.
	 */
	boolean isOpenTransaction(Transaction transaction) throws PersistenceException;
	
	/**
	 * This Method creates an instance of a given User Type. The User Type has to be a concrete Type, otherwise an
	 * exception will be thrown. If the the type is also a user transaction, a new transaction will be opened.
	 * 
	 * @param type
	 *            A User Type which is not abstract.
	 * @param transaction
	 *            A Transaction in which scope the instance will be created.
	 * @return Provides the new Instance to type.
	 * @throws PersistenceException
	 *             This exception occurs if there are any problems contacting the database.
	 */
	UserObject create(UserType type, Transaction transaction) throws PersistenceException;
	
	/**
	 * This Method deletes a given User Object in the scope of transaction transaction. This method only works if there
	 * are Links anymore, where object is owner or target of the link (defensive manner).
	 * 
	 * @param object
	 *            A User Objects, which is accessible in the scope of transaction and there are links anymore pointing
	 *            to.
	 * @param transaction
	 *            A currently open transaction.
	 * @throws PersistenceException
	 *             This exception occurs if there are any problems contacting the database.
	 */
	void delete(UserObject object, Transaction transaction) throws PersistenceException;
	
	// ---------- Set ----------
	
	/**
	 * This method creates a link between the user Object owner and the user object target. If the association is
	 * essential a modified flag from transaction will be set for owner. Also if the association is unique, the previos
	 * existing link will be deleted. The UnidirectionalLink will be an instance of the UnidirectionalAssociation
	 * association. The owner has to be a subtype of association.owner and the target has to be a subtype of
	 * association.target otherwise an exception will be thrown. Also owner and target have to be in scope of
	 * transaction. The link also will be created in scope of transaction.
	 * 
	 * @param owner
	 *            A User Object, where type <= a.owner
	 * @param association
	 *            An UnidirectionalAssociation
	 * @param target
	 *            A User Object, where type <= a.target
	 * @param transaction
	 *            A currently open Transaction.
	 * @return Provides a new UnidirectionalLink, which is an instance of a.
	 * @throws PersistenceException
	 *             This exception occurs if there are any problems contacting the database or there are consistencies
	 *             violated.
	 */
	UnidirectionalLink set(UserObject owner,
			UnidirectionalAssociation association,
			UserObject target,
			Transaction transaction) throws PersistenceException;
	
	/**
	 * This method creates a link between the user Object owner and the user object target. If the association is
	 * essential a modified flag from transaction will be set for owner. Also if the association is unique, the previos
	 * existing link will be deleted. The UnidirectionalLink will be an instance of the UnidirectionalAssociation
	 * association. The owner has to be a subtype of association.owner and the target has to be a subtype of
	 * association.target otherwise an exception will be thrown. Also owner and target have to be in scope of
	 * transaction. The link also will be created in scope of transaction.
	 * 
	 * @param owner
	 *            A User Object, where type <= a.owner
	 * @param association
	 *            An UnidirectionalAssociation
	 * @param target
	 *            An Integer Value
	 * @param transaction
	 *            A currently open Transaction.
	 * @return Provides a new UnidirectionalLink, which is an instance of a.
	 * @throws PersistenceException
	 *             This exception occurs if there are any problems contacting the database or there consistencies are
	 *             violated.
	 */
	UnidirectionalLink set(UserObject owner,
			UnidirectionalAssociation association,
			BigInteger target,
			Transaction transaction) throws PersistenceException;
	
	/**
	 * This method creates a link between the user Object owner and the user object target. If the association is
	 * essential a modified flag from transaction will be set for owner. Also if the association is unique, the previos
	 * existing link will be deleted. The UnidirectionalLink will be an instance of the UnidirectionalAssociation
	 * association. The owner has to be a subtype of association.owner and the target has to be a subtype of
	 * association.target otherwise an exception will be thrown. Also owner and target have to be in scope of
	 * transaction. The link also will be created in scope of transaction.
	 * 
	 * @param owner
	 *            A User Object, where type <= a.owner
	 * @param association
	 *            An UnidirectionalAssociation
	 * @param target
	 *            An String Value.
	 * @param transaction
	 *            A currently open Transaction.
	 * @return Provides a new UnidirectionalLink, which is an instance of a.
	 * @throws PersistenceException
	 *             This exception occurs if there are any problems contacting the database or there are consistencies
	 *             violated.
	 */
	UnidirectionalLink set(UserObject owner,
			UnidirectionalAssociation association,
			String target,
			Transaction transaction) throws PersistenceException;
	
	// ---------- Put ----------
	
	/**
	 * This method creates a link between the user Object owner, the user object target and the map key. If the
	 * association is essential a modified flag from t will be set for owner. The previos existing link will be deleted.
	 * The MapLink will be an instance of the MapAssociation association. The owner has to be a subtype of
	 * association.owner, the target has to be a subtype of association.target and the key has to be a subtype of
	 * association.keyType otherwise an exception will be thrown. Also owner, target and key have to be in scope of
	 * transaction. The link also will be created in scope of transaction.
	 * 
	 * @param owner
	 *            A User Object, where type <= a.owner
	 * @param association
	 *            An MapAssociation
	 * @param target
	 *            A User Object, where type <= a.target
	 * @param key
	 *            The key-Object
	 * @param transaction
	 *            A currently open Transaction.
	 * @return Provides a new MapLink, which is an instance of association.
	 * @throws PersistenceException
	 *             This exception occurs if there are any problems contacting the database or there are consistencies
	 *             violated.
	 */
	MapLink put(UserObject owner, MapAssociation association, UserObject target, String key, Transaction transaction)
			throws PersistenceException;
	
	/**
	 * This method creates a link between the user Object owner, the user object target and the map key. If the
	 * association is essential a modified flag from t will be set for owner. The previos existing link will be deleted.
	 * The MapLink will be an instance of the MapAssociation association. The owner has to be a subtype of
	 * association.owner, the target has to be a subtype of association.target and the key has to be a subtype of
	 * association.keyType otherwise an exception will be thrown. Also owner, target and key have to be in scope of
	 * transaction. The link also will be created in scope of transaction.
	 * 
	 * @param owner
	 *            A User Object, where type <= a.owner
	 * @param association
	 *            An MapAssociation
	 * @param target
	 *            An Integer Value
	 * @param key
	 *            The key-Object
	 * @param transaction
	 *            A currently open Transaction.
	 * @return Provides a new MapLink, which is an instance of association.
	 * @throws PersistenceException
	 *             This exception occurs if there are any problems contacting the database or there consistencies are
	 *             violated.
	 */
	MapLink put(UserObject owner, MapAssociation association, BigInteger target, String key, Transaction transaction)
			throws PersistenceException;
	
	/**
	 * This method creates a link between the user Object owner, the user object target and the map key. If the
	 * association is essential a modified flag from t will be set for owner. The previos existing link will be deleted.
	 * The MapLink will be an instance of the MapAssociation association. The owner has to be a subtype of
	 * association.owner, the target has to be a subtype of association.target and the key has to be a subtype of
	 * association.keyType otherwise an exception will be thrown. Also owner, target and key have to be in scope of
	 * transaction. The link also will be created in scope of transaction.
	 * 
	 * @param owner
	 *            A User Object, where type <= a.owner
	 * @param association
	 *            An MapAssociation
	 * @param target
	 *            An String Value.
	 * @param key
	 *            The key-Object
	 * @param transaction
	 *            A currently open Transaction.
	 * @return Provides a new MapLink, which is an instance of association.
	 * @throws PersistenceException
	 *             This exception occurs if there are any problems contacting the database or there are consistencies
	 *             violated.
	 */
	MapLink put(UserObject owner, MapAssociation association, String target, String key, Transaction transaction)
			throws PersistenceException;
	
	/**
	 * This method creates a link between the user Object owner, the user object target and the map key. If the
	 * association is essential a modified flag from t will be set for owner. The previos existing link will be deleted.
	 * The MapLink will be an instance of the MapAssociation association. The owner has to be a subtype of
	 * association.owner, the target has to be a subtype of association.target and the key has to be a subtype of
	 * association.keyType otherwise an exception will be thrown. Also owner, target and key have to be in scope of
	 * transaction. The link also will be created in scope of transaction.
	 * 
	 * @param owner
	 *            A User Object, where type <= a.owner
	 * @param association
	 *            An MapAssociation
	 * @param target
	 *            A User Object, where type <= a.target
	 * @param key
	 *            The key-Object
	 * @param transaction
	 *            A currently open Transaction.
	 * @return Provides a new MapLink, which is an instance of association.
	 * @throws PersistenceException
	 *             This exception occurs if there are any problems contacting the database or there are consistencies
	 *             violated.
	 */
	MapLink put(UserObject owner, MapAssociation association, UserObject target, BigInteger key, Transaction transaction)
			throws PersistenceException;
	
	/**
	 * This method creates a link between the user Object owner, the user object target and the map key. If the
	 * association is essential a modified flag from t will be set for owner. The previos existing link will be deleted.
	 * The MapLink will be an instance of the MapAssociation association. The owner has to be a subtype of
	 * association.owner, the target has to be a subtype of association.target and the key has to be a subtype of
	 * association.keyType otherwise an exception will be thrown. Also owner, target and key have to be in scope of
	 * transaction. The link also will be created in scope of transaction.
	 * 
	 * @param owner
	 *            A User Object, where type <= a.owner
	 * @param association
	 *            An MapAssociation
	 * @param target
	 *            An Integer Value
	 * @param key
	 *            The key-Object
	 * @param transaction
	 *            A currently open Transaction.
	 * @return Provides a new MapLink, which is an instance of association.
	 * @throws PersistenceException
	 *             This exception occurs if there are any problems contacting the database or there consistencies are
	 *             violated.
	 */
	MapLink put(UserObject owner, MapAssociation association, BigInteger target, BigInteger key, Transaction transaction)
			throws PersistenceException;
	
	/**
	 * This method creates a link between the user Object owner, the user object target and the map key. If the
	 * association is essential a modified flag from t will be set for owner. The previos existing link will be deleted.
	 * The MapLink will be an instance of the MapAssociation association. The owner has to be a subtype of
	 * association.owner, the target has to be a subtype of association.target and the key has to be a subtype of
	 * association.keyType otherwise an exception will be thrown. Also owner, target and key have to be in scope of
	 * transaction. The link also will be created in scope of transaction.
	 * 
	 * @param owner
	 *            A User Object, where type <= a.owner
	 * @param association
	 *            An MapAssociation
	 * @param target
	 *            An String Value.
	 * @param key
	 *            The key-Object
	 * @param transaction
	 *            A currently open Transaction.
	 * @return Provides a new MapLink, which is an instance of association.
	 * @throws PersistenceException
	 *             This exception occurs if there are any problems contacting the database or there are consistencies
	 *             violated.
	 */
	MapLink put(UserObject owner, MapAssociation association, String target, BigInteger key, Transaction transaction)
			throws PersistenceException;
	
	/**
	 * This method creates a link between the user Object owner, the user object target and the map key. If the
	 * association is essential a modified flag from t will be set for owner. The previos existing link will be deleted.
	 * The MapLink will be an instance of the MapAssociation association. The owner has to be a subtype of
	 * association.owner, the target has to be a subtype of association.target and the key has to be a subtype of
	 * association.keyType otherwise an exception will be thrown. Also owner, target and key have to be in scope of
	 * transaction. The link also will be created in scope of transaction.
	 * 
	 * @param owner
	 *            A User Object, where type <= a.owner
	 * @param association
	 *            An MapAssociation
	 * @param target
	 *            A User Object, where type <= a.target
	 * @param key
	 *            The key-Object
	 * @param transaction
	 *            A currently open Transaction.
	 * @return Provides a new MapLink, which is an instance of association.
	 * @throws PersistenceException
	 *             This exception occurs if there are any problems contacting the database or there are consistencies
	 *             violated.
	 */
	MapLink put(UserObject owner, MapAssociation association, UserObject target, UserObject key, Transaction transaction)
			throws PersistenceException;
	
	/**
	 * This method creates a link between the user Object owner, the user object target and the map key. If the
	 * association is essential a modified flag from t will be set for owner. The previos existing link will be deleted.
	 * The MapLink will be an instance of the MapAssociation association. The owner has to be a subtype of
	 * association.owner, the target has to be a subtype of association.target and the key has to be a subtype of
	 * association.keyType otherwise an exception will be thrown. Also owner, target and key have to be in scope of
	 * transaction. The link also will be created in scope of transaction.
	 * 
	 * @param owner
	 *            A User Object, where type <= a.owner
	 * @param association
	 *            An MapAssociation
	 * @param target
	 *            An Integer Value
	 * @param key
	 *            The key-Object
	 * @param transaction
	 *            A currently open Transaction.
	 * @return Provides a new MapLink, which is an instance of association.
	 * @throws PersistenceException
	 *             This exception occurs if there are any problems contacting the database or there consistencies are
	 *             violated.
	 */
	MapLink put(UserObject owner, MapAssociation association, BigInteger target, UserObject key, Transaction transaction)
			throws PersistenceException;
	
	/**
	 * This method creates a link between the user Object owner, the user object target and the map key. If the
	 * association is essential a modified flag from t will be set for owner. The previos existing link will be deleted.
	 * The MapLink will be an instance of the MapAssociation association. The owner has to be a subtype of
	 * association.owner, the target has to be a subtype of association.target and the key has to be a subtype of
	 * association.keyType otherwise an exception will be thrown. Also owner, target and key have to be in scope of
	 * transaction. The link also will be created in scope of transaction.
	 * 
	 * @param owner
	 *            A User Object, where type <= a.owner
	 * @param association
	 *            An MapAssociation
	 * @param target
	 *            An String Value.
	 * @param key
	 *            The key-Object
	 * @param transaction
	 *            A currently open Transaction.
	 * @return Provides a new MapLink, which is an instance of association.
	 * @throws PersistenceException
	 *             This exception occurs if there are any problems contacting the database or there are consistencies
	 *             violated.
	 */
	MapLink put(UserObject owner, MapAssociation association, String target, UserObject key, Transaction transaction)
			throws PersistenceException;
	
	// ---------- Other ----------
	
	/**
	 * This method deletes a given Link in the scope of the transaction transaction. If the type of the link has an
	 * essential flag an modified flag for the owner of the Link will be set.
	 * 
	 * @param link
	 *            A UnidirectionalLink which will be deleted.
	 * @param transaction
	 *            A currently open transaction.
	 * @throws PersistenceException
	 *             This exception occurs if there are any problems contacting the database.
	 */
	void unset(Link link, Transaction transaction) throws PersistenceException;
	
	/**
	 * This Methods commits a currently open transaction, so all modification transaction has done will be accessible to
	 * all other transaction and can be seen in future revisions. If there are other open Transactions transaction',
	 * where isInConflict(transaction',transaction) = true, then each transaction' will be rolled back.
	 * 
	 * @param transaction
	 *            A currently open transaction.
	 * @throws PersistenceException
	 *             This exception occurs if there are any problems contacting the database.
	 */
	void commit(Transaction transaction) throws PersistenceException;
	
	/**
	 * This methods rolls a transaction back, this means that all modifications of transaction will be removed. If
	 * transaction is an adhoc Transaction transaction will also be closed.
	 * 
	 * @param transaction
	 *            A currently open Transaction.
	 * @throws PersistenceException
	 *             This exception occurs if there are any problems contacting the database.
	 */
	void rollback(Transaction transaction) throws PersistenceException;
	
	/**
	 * This Methods creates a save point inside a transaction. So modifications of transaction until now are saved so it
	 * is possible to rollback to this save point later.
	 * 
	 * @param transaction
	 *            A currently open transaction.
	 * @throws PersistenceException
	 *             This exception occurs if there are any problems contacting the database.
	 */
	void savePoint(Transaction transaction) throws PersistenceException;
	
	/**
	 * This Method deletes all modifications of transaction since the last created save point of transaction. If
	 * transaction has never created a save point all modifications will be lost.
	 * 
	 * @param transaction
	 *            A currently open transaction.
	 * @throws PersistenceException
	 *             This exception occurs if there are any problems contacting the database.
	 */
	void rollbackToSavePoint(Transaction transaction) throws PersistenceException;
	
	/**
	 * This Method provides an Integer Value with the given Id. If there is no Integer value with the given id, an
	 * exception will be thrown.
	 * 
	 * @param id
	 *            An id of an integer value inside the database.
	 * @return Provides a Big Integer Value Object.
	 * @throws PersistenceException
	 *             This exception occurs if there are any problems contacting the database or there is not Integer Value
	 *             with the given id.
	 */
	BigInteger getIntForId(long id) throws PersistenceException;
	
	/**
	 * This Method provides a String Value with the given Id. If there is no String value with the given id, an
	 * exception will be thrown.
	 * 
	 * @param id
	 *            An id of a string value inside the database.
	 * @return Provides a String Value Object.
	 * @throws PersistenceException
	 *             This exception occurs if there are any problems contacting the database or there is no String with
	 *             the given id.
	 */
	String getStringForId(long id) throws PersistenceException;
	
	/**
	 * Provides the id of a given String value string. If there is no String value same as string is present inside the
	 * database a new String Value will be created in the database.
	 * 
	 * @param string
	 *            A String s
	 * @return The Id of s in the database.
	 * @throws PersistenceException
	 *             This exception occurs if there are any problems contacting the database.
	 */
	long getIdForString(String string) throws PersistenceException;
	
	/**
	 * Provides the id of a given Integer value integer. If there is no Integer value same as is is present inside the
	 * database a new Integer value will be created in the database.
	 * 
	 * @param integer
	 *            An Integer i
	 * @return The Id of i in the database.
	 * @throws PersistenceException
	 *             This exception occurs if there are any problems contacting the database.
	 */
	long getIdForInteger(BigInteger integer) throws PersistenceException;
	
	/**
	 * This Method provides a new Adhoc Transaction, which can be used to manipulate data.
	 * 
	 * @return A new open Adoc Transaction.
	 * @throws PersistenceException
	 *             This exception occurs if there are any problems contacting the database.
	 */
	Transaction provideAdhocTransaction() throws PersistenceException;
	
	/**
	 * Provides an instance of the TypeManager, an object which has knowledge about every type and association in the
	 * knowledge-tier. Before you cann call this operation, you have to initialize this facade.
	 * 
	 * @return An Instance of the Type Manager.
	 */
	TypeManager getTypeManager();
	
	/**
	 * !! ATTENTION !!! This method clears the object world completely, therefore all data will be lost. So use this
	 * method carefully and only during development time !!!
	 * 
	 * @throws PersistenceException
	 *             This exception occurs if there are any problems contacting the database.
	 */
	void clear() throws PersistenceException;
	
	/**
	 * This method provides all instances for a given User Type, which can be seen in the current transaction. Because
	 * the provided dataset may be large, this method created no read-flags on the returned objects, so it is possible
	 * the run into race-conditions on the returned objects.
	 * 
	 * @param type
	 *            The Type, which the reurned objects are instances of.
	 * @param transaction
	 *            The current Transaction, which is used
	 * @return A Collection of User Types, which can be seen at the moment and are instances of the given user type.
	 * @throws PersistenceException
	 *             This exception occurs if there are any problems contacting the database.
	 */
	Collection<UserObject> findAllObjects(UserType type, Transaction transaction) throws PersistenceException;
	
	/**
	 * This provides the UserObject with the given id and marks the corresponding Object as read, so it will be under
	 * transaction control.
	 * 
	 * @param id
	 *            The Id of the Provided User Object.
	 * @param transaction
	 *            The Transaction which will place the read marker on the object.
	 * @return Provides a User Object with the given Id.
	 * @throws PersistenceException
	 *             An Exception will be thrown if something went wrong.
	 */
	UserObject checkUserObjectOut(long id, Transaction transaction) throws PersistenceException;
	
}
