package de.fhdw.wtf.persistence.facade;

import java.sql.SQLException;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import de.fhdw.wtf.persistence.exception.ClassFacadeUninitializedException;
import de.fhdw.wtf.persistence.exception.DuplicateUserTypeException;
import de.fhdw.wtf.persistence.exception.NotDeletableException;
import de.fhdw.wtf.persistence.exception.NotValidOperationException;
import de.fhdw.wtf.persistence.exception.PersistenceException;
import de.fhdw.wtf.persistence.exception.SpecializationCycleDetected;
import de.fhdw.wtf.persistence.exception.TypeOrAssociationNotFoundException;
import de.fhdw.wtf.persistence.meta.Association;
import de.fhdw.wtf.persistence.meta.IntegerType;
import de.fhdw.wtf.persistence.meta.MapAssociation;
import de.fhdw.wtf.persistence.meta.Specialization;
import de.fhdw.wtf.persistence.meta.StringType;
import de.fhdw.wtf.persistence.meta.Type;
import de.fhdw.wtf.persistence.meta.TypeVisitorException;
import de.fhdw.wtf.persistence.meta.UnidirectionalAssociation;
import de.fhdw.wtf.persistence.meta.UserType;

/**
 * A class to represent an implementation of the ClassFacade Interface for the usage without a database.
 */
public class NoDatabaseClassFacadeImplementation implements ClassFacade {
	
	/**
	 * Represents the id which will be given to the first thing with an identifier which has been created by this
	 * ClassFacade. Will be set during the ClassFacades instantiation or during an execution of the clear-operation.
	 */
	private static final long INITIAL_NEXT_ID = 3;
	
	/**
	 * Message for the NotValidOperationException in regard to the attempt of renaming a base-type.
	 */
	private static final String RENAMING_BASE_TYPE_MESSAGE = "Renaming a BaseType is not supported!";
	
	/**
	 * The Type Manager which stores the Model Item informations.
	 */
	private final TypeManagerImplementation typeManager;
	
	/**
	 * This field contains the next identifier for something that can be created within this ClassFacade.
	 */
	private long nextId;
	
	/**
	 * Creates a new class-facade for usage with the no-database.
	 */
	public NoDatabaseClassFacadeImplementation() {
		this.typeManager = TypeManagerImplementation.getInstance();
		this.initializeBaseTypes();
		this.nextId = NoDatabaseClassFacadeImplementation.INITIAL_NEXT_ID;
	}
	
	@Override
	public UserType createUserType(final String name, final boolean abs, final boolean transaction)
			throws DuplicateUserTypeException {
		try {
			this.typeManager.getTypeforName(name);
			throw new DuplicateUserTypeException(name);
		} catch (final TypeOrAssociationNotFoundException e) {
			final UserType userType = new UserType(this.getAndIncrementNextId(), name, abs, transaction);
			this.typeManager.saveType(userType);
			this.typeManager.saveSpecialization(new Specialization(userType, userType));
			return userType;
		}
	}
	
	@Override
	public UnidirectionalAssociation createUnidirectionalAssociation(final String name,
			final boolean essential,
			final boolean unique,
			final UserType owner,
			final Type target) throws PersistenceException {
		final UnidirectionalAssociation association =
				new UnidirectionalAssociation(this.getAndIncrementNextId(), name, owner, target, essential, unique);
		this.typeManager.saveAssociation(association);
		return association;
	}
	
	@Override
	public MapAssociation createMapAssociation(final String name,
			final boolean essential,
			final UserType owner,
			final Type target,
			final Type keyType) throws PersistenceException {
		final MapAssociation association =
				new MapAssociation(this.getAndIncrementNextId(), name, owner, target, keyType, essential);
		this.typeManager.saveAssociation(association);
		return association;
	}
	
	@Override
	public void createSpecializationBetween(final UserType ancestor, final Type descendant) throws PersistenceException {
		final Collection<Specialization> allSpecializations = this.typeManager.getAllSpecializations();
		if (allSpecializations.contains(new Specialization(descendant, ancestor))) {
			throw new SpecializationCycleDetected(new SQLException());
		}
		final Iterator<Specialization> allSpecializationsIterator = allSpecializations.iterator();
		while (allSpecializationsIterator.hasNext()) {
			final Specialization current = allSpecializationsIterator.next();
			final boolean notReflexive = !current.getAncestor().isTheSameAs(current.getDescendant());
			if (notReflexive && ancestor.isTheSameAs(current.getDescendant())) {
				this.typeManager.saveSpecialization(new Specialization(current.getAncestor(), descendant));
			} else if (notReflexive && descendant.isTheSameAs(current.getAncestor())) {
				this.typeManager.saveSpecialization(new Specialization(ancestor, current.getDescendant()));
			}
		}
		this.typeManager.saveSpecialization(new Specialization(ancestor, descendant));
	}
	
	@Override
	public boolean isSuperClassTo(final Type ancestor, final Type descendant) throws PersistenceException {
		final Iterator<Specialization> allSpecializations = this.typeManager.getAllSpecializations().iterator();
		while (allSpecializations.hasNext()) {
			final Specialization current = allSpecializations.next();
			if (current.getAncestor().isTheSameAs(ancestor) && current.getDescendant().isTheSameAs(descendant)) {
				return true;
			}
		}
		return false;
	}
	
	@Override
	public void finalizeSpecialization() throws PersistenceException {
		// No need to finalize the specialization because the transitive closure is ensured by the create method for
		// specializations while the reflexive closure is ensured by the create method for UserTypes.
	}
	
	@Override
	public void clear() throws PersistenceException {
		this.typeManager.clear();
		this.nextId = NoDatabaseClassFacadeImplementation.INITIAL_NEXT_ID;
	}
	
	@Override
	public void renameType(final Long typeId, final String newName) throws PersistenceException {
		final Type type = this.getTypeManager().getTypeForId(typeId);
		type.accept(new TypeVisitorException<PersistenceException>() {
			
			@Override
			public void handleUserType(final UserType argument) throws PersistenceException {
				NoDatabaseClassFacadeImplementation.this.deleteUserType(typeId);
				NoDatabaseClassFacadeImplementation.this.typeManager.saveType(new UserType(argument.getId(), newName,
						argument.isAbs(), argument.isTrans()));
			}
			
			@Override
			public void handleBaseType(final Type t) throws NotValidOperationException {
				throw new NotValidOperationException(NoDatabaseClassFacadeImplementation.RENAMING_BASE_TYPE_MESSAGE);
			}
		});
	}
	
	@Override
	public void renameAssociation(final Long assoId, final String newName) throws PersistenceException {
		try {
			final UnidirectionalAssociation association = this.typeManager.getUnidirectionalAssociationForId(assoId);
			this.deleteAssociation(assoId);
			this.typeManager.saveAssociation(new UnidirectionalAssociation(association.getId(), newName, association
					.getOwner(), association.getTarget(), association.isEssential(), association.isUnique()));
		} catch (final TypeOrAssociationNotFoundException exception) {
			final MapAssociation association = this.typeManager.getMapAssociationForId(assoId);
			this.deleteAssociation(assoId);
			this.typeManager.saveAssociation(new MapAssociation(association.getId(), newName, association.getOwner(),
					association.getTarget(), association.getKeyType(), association.isEssential()));
		}
	}
	
	@Override
	public void deleteAssociation(final Long associationId) throws PersistenceException {
		this.typeManager.deleteAssociation(associationId);
	}
	
	@Override
	public void deleteUserType(final Long typeId) throws PersistenceException {
		final Type type = this.getTypeManager().getTypeForId(typeId);
		type.accept(new TypeVisitorException<NotDeletableException>() {
			@Override
			public void handleUserType(final UserType argument) throws NotDeletableException {
				try {
					NoDatabaseClassFacadeImplementation.this.typeManager.deleteType(typeId);
					final Iterator<Specialization> allSpecializations =
							NoDatabaseClassFacadeImplementation.this.typeManager.getAllSpecializations().iterator();
					while (allSpecializations.hasNext()) {
						final Specialization current = allSpecializations.next();
						if (current.getAncestor().isTheSameAs(argument)
								|| current.getDescendant().isTheSameAs(argument)) {
							NoDatabaseClassFacadeImplementation.this.typeManager.deleteSpecialization(current);
						}
					}
				} catch (final TypeOrAssociationNotFoundException e) {
					// Should not happen because the type was already found.
					throw new Error();
				}
			}
			
			@Override
			public void handleBaseType(final Type t) throws NotDeletableException {
				throw new NotDeletableException();
			}
		});
	}
	
	@Override
	public void updateLinksToNewAssociation(final Long associationId, final Collection<Long> newAssociationIds)
			throws PersistenceException {
		
		// -------------------------------------------------------------
		// TODO Auto-generated method stub
		// It seems to be a refactoring method.
		// -------------------------------------------------------------
		
		// FOR i IN 1 .. newAssociations.COUNT
		// LOOP
		// UPDATE Link l SET l.instance_of=newAssociations (i)
		// WHERE l.instance_of=oldAssociation AND
		// ISSuperClassTo(
		// (SELECT a.owner FROM Association a
		// WHERE a.id=newAssociations (i)),
		// (SELECT o.instance_of FROM Object o
		// WHERE o.id=l.owner)
		// ) > 0;
		// END LOOP;
		
		// final ObjectFacade objectFacade = NoDatabaseManager.getInstance().getObjectFacade();
		// final Iterator<Long> newAssociationIdsIterator = newAssociationIds.iterator();
		// while (newAssociationIdsIterator.hasNext()) {
		// // TODO
		// }
	}
	
	@Override
	public void moveLinksAndCreateObjects(final List<Long> oldAssoIds,
			final Association newAsso,
			final UserType newType,
			final List<Long> newAssoIds) throws PersistenceException {
		// TODO Auto-generated method stub
		// It seems to be a refactoring method.
		// It is still not implemented in the oracle database.
	}
	
	/**
	 * Adds all BaseTypes and there respective reflexive specialization-pairs to the TypeManager of this ClassFacade.
	 */
	private void initializeBaseTypes() {
		this.typeManager.saveType(StringType.getInstance());
		this.typeManager.saveSpecialization(new Specialization(StringType.getInstance(), StringType.getInstance()));
		this.typeManager.saveType(IntegerType.getInstance());
		this.typeManager.saveSpecialization(new Specialization(IntegerType.getInstance(), IntegerType.getInstance()));
	}
	
	@Override
	public void initialize() throws PersistenceException {
		// Initialization measurements are not needed in this implementation because model informations are not being
		// saved longer than one runtime-session.
	}
	
	@Override
	public void initializeForRuntime() throws PersistenceException {
		// Initialization measurements are not needed in this implementation because model informations are not being
		// saved longer than one runtime-session.
	}
	
	@Override
	public boolean hasBeenInitialized() {
		// No further initialization is needed in this implementation and thus every instance of this ClassFacade is
		// already initialized after its constructor-call.
		return true;
	}
	
	@Override
	public TypeManager getTypeManager() throws ClassFacadeUninitializedException {
		return this.typeManager;
	}
	
	/**
	 * @return Returns the next free id.
	 */
	private long getAndIncrementNextId() {
		final long result = this.nextId;
		this.nextId++;
		return result;
	}
	
}
