package de.fhdw.wtf.persistence.meta;

import java.math.BigInteger;

import de.fhdw.wtf.persistence.exception.PersistenceException;
import de.fhdw.wtf.persistence.facade.ObjectFacade;

/**
 * A class to represent any Integer Value in the database. Intger Values are instances of Integer Type.
 * 
 */
public class IntegerValue extends Object { // NOPMD: Object != java.lang.Object

	/**
	 * If there are values which are not already fetched the database is asked.
	 */
	private static ObjectFacade facade;
	
	/**
	 * set the ObjectFacade, because when there are values which are not already fetched the database is asked.
	 * 
	 * @param Objectfacade
	 */
	public static void setObjectFacade(final ObjectFacade objectFacade) {
		facade = objectFacade;
	}
	
	/**
	 * The value of the integer.
	 */
	private BigInteger value;
	
	/**
	 * The id of the integer in the database.
	 */
	private Long id;
	
	/**
	 * Constructor for a new integer value. This constructor could be used if only the id is knwown. The value will be
	 * fetched from the database.
	 * 
	 * @param id
	 *            A valid id of an integer value, which is present in the database.
	 */
	public IntegerValue(final long id) {
		super(IntegerType.getInstance());
		this.id = id;
	}
	
	/**
	 * Constructor for a new integer value. This constructor could be used if only a value is known. The id in the
	 * database will be determined automatically.
	 * 
	 * @param value
	 *            An integer value.
	 */
	public IntegerValue(final BigInteger value) {
		super(IntegerType.getInstance());
		this.value = value;
	}
	
	/**
	 * Constructor for an integer value if id and value are knwown.
	 * 
	 * @param id
	 *            The id of the integer value in the database.
	 * @param value
	 *            The value of the integer value.
	 */
	public IntegerValue(final Long id, final BigInteger value) {
		super(IntegerType.getInstance());
		this.id = id;
		this.value = value;
	}
	
	/**
	 * Getter for value. Lazy loads the value from the database if its unset.
	 * 
	 * @return Provides the value of this Integer Value Object.
	 */
	public BigInteger getValue() {
		if (this.value == null) {
			try {
				this.value = facade.getIntForId(this.getId());
			} catch (final PersistenceException e) {
				throw new Error(e);
			}
		}
		return this.value;
	}
	
	/**
	 * Getter for Id. Lazy loads the Id from the database if unset.
	 */
	@Override
	public long getId() {
		if (this.id == null) {
			try {
				this.id = facade.getIdForInteger(this.getValue());
			} catch (final PersistenceException e) {
				throw new Error(e);
			}
		}
		return this.id;
	}
	
	@Override
	public boolean isTheSameAs(final java.lang.Object other) {
		if (!(other instanceof IntegerValue)) {
			return false;
		}
		
		return this.getId() == ((IntegerValue) other).getId();
	}
	
}
