package de.fhdw.wtf.persistence.meta;

/**
 * A class to represent a connection between two objects in the database. The UnidirectionalLink itself is an instance
 * of an association.
 * 
 */
public abstract class Link extends InstanceItem {
	
	/**
	 * The identity of the link in the database. This Id is unique.
	 */
	private final long id;
	
	/**
	 * The owning User Object of the link.
	 */
	private final UserObject owner;
	
	/**
	 * The targeted Object.
	 */
	private final Object target;
	
	/**
	 * Provides the Type of this UnidirectionalLink.
	 * 
	 * @return Provides the association for which this link is an instance.
	 */
	public abstract Association getInstanceOf();
	
	/**
	 * Getter for the Owner of this UnidirectionalLink.
	 * 
	 * @return Provides the User Object, which is the owner of this link.
	 */
	public UserObject getOwner() {
		return this.owner;
	}
	
	/**
	 * Getter for the target of this UnidirectionalLink.
	 * 
	 * @return Provides an Object, which is the target of this link.
	 */
	public Object getTarget() {
		return this.target;
	}
	
	/**
	 * Constructor for a new UnidirectionalLink, it should only be called by the Object Facade or Junit Test to
	 * guarantee a valid consistency with the database.
	 * 
	 * @param id
	 *            The Id of the UnidirectionalLink, determined by the database.
	 * @param owner
	 *            The owner of the this UnidirectionalLink. A User Object.
	 * @param target
	 *            The target of this link. An Object.
	 */
	public Link(final long id, final UserObject owner, final Object target) {
		super();
		this.id = id;
		this.owner = owner;
		this.target = target;
	}
	
	@Override
	public long getId() {
		return this.id;
	}
	
	@Override
	public boolean equals(final java.lang.Object obj) {
		if (!(obj instanceof Link)) {
			return false;
		}
		
		final Link link = (Link) obj;
		return this.getOwner().equals(link.getOwner()) && this.getTarget().equals(link.getTarget());
	}
	
	@Override
	public int hashCode() {
		return this.owner.hashCode() ^ this.target.hashCode();
	}
	
	@Override
	public String toString() {
		return "UnidirectionalLink<" + this.getId() + ">";
	}
	
	@Override
	public boolean isTheSameAs(final java.lang.Object other) {
		if (!(other instanceof Link)) {
			return false;
		}
		
		return this.getId() == ((Link) other).getId();
	}
	
}
