package de.fhdw.wtf.persistence.utils;

import java.util.HashMap;
import java.util.MissingResourceException;

/**
 * This class is a singleton that is used to create a PropertiesReader class that stores the connection properties to
 * the oracle database.
 * 
 * @author HFW413hy
 *
 */
public class PropertiesReaderFactory {
	
	private static PropertiesReaderFactory _instance = null;
	
	/**
	 * Returns the instance of this singleton.
	 * 
	 * @return The instance of this singleton.
	 */
	public static synchronized PropertiesReaderFactory getInstance() {
		if (_instance == null) {
			_instance = new PropertiesReaderFactory();
		}
		return _instance;
	}
	
	/**
	 * Creates a PropertiesReaderFile-Object that will be returned.
	 * 
	 * @param file
	 * @return The PropertiesReaderFile-Object
	 */
	public PropertiesReader getPropertiesReaderFile(final String file) {
		final PropertiesReader reader = new PropertiesReaderFile();
		reader.initialize(file);
		return reader;
	}
	
	/**
	 * Returns a new PropertiesReader that stores the Oracle connection properties. At first, it searches for a file
	 * "oracle.properties" in the current directory. If there is no file that could be read, it tries to read the
	 * "oracle.properties" from the current namespace. If there is not file, it will return a PropertiesReader that
	 * contains default information.
	 * 
	 * @return The PropertiesReader that contains connection properties to the oracle database.
	 */
	public PropertiesReader getPropertiesReaderOracle() {
		PropertiesReader reader;
		
		// Try to read the local file from the current directory
		// --------------------------------------------------
		reader = new PropertiesReaderFile();
		try {
			reader.initialize("oracle.properties");
			return reader;
		} catch (final MissingResourceException e) {
			// Continue if this happens... file is not here
		}
		
		// Try to read the ressource file from the current package
		// --------------------------------------------------
		reader = new PropertiesReaderRessource();
		try {
			reader.initialize("oracle");
			return reader;
		} catch (final MissingResourceException e) {
			// Continue if this happens... file is not here
		}
		
		// Now, the default reader with default params
		// --------------------------------------------------
		reader = new PropertiesReader() {
			private final HashMap<String, String> propValues = new HashMap<>();
			
			@Override
			public void initialize(final String propFile) throws MissingResourceException {
				this.propValues.put("sid", "XE");
				this.propValues.put("schema", "meta");
				this.propValues.put("username", "meta");
				this.propValues.put("password", "meta");
				this.propValues.put("hostname", "localhost");
				this.propValues.put("port", "1521");
			}
			
			@Override
			public String getProperty(final String key) {
				if (!this.propValues.containsKey(key)) {
					return "";
				}
				return this.propValues.get(key);
			}
		};
		
		return reader;
	}
}
