package de.fhdw.wtf.persistence.facade;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.sql.CallableStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashSet;

import oracle.jdbc.OracleCallableStatement;
import oracle.jdbc.OracleTypes;
import de.fhdw.wtf.persistence.exception.BaseTypeNotFoundException;
import de.fhdw.wtf.persistence.exception.DuplicateEntryException;
import de.fhdw.wtf.persistence.exception.InvalidLinkException;
import de.fhdw.wtf.persistence.exception.NoValidTransactionException;
import de.fhdw.wtf.persistence.exception.NotInScopeException;
import de.fhdw.wtf.persistence.exception.NotInstantiatableException;
import de.fhdw.wtf.persistence.exception.ObjectNotFoundException;
import de.fhdw.wtf.persistence.exception.OtherSQLException;
import de.fhdw.wtf.persistence.exception.PersistenceException;
import de.fhdw.wtf.persistence.exception.ToManyObjectsException;
import de.fhdw.wtf.persistence.meta.AdhocTransaction;
import de.fhdw.wtf.persistence.meta.Association;
import de.fhdw.wtf.persistence.meta.IntegerValue;
import de.fhdw.wtf.persistence.meta.Link;
import de.fhdw.wtf.persistence.meta.MapAssociation;
import de.fhdw.wtf.persistence.meta.MapLink;
import de.fhdw.wtf.persistence.meta.Object;
import de.fhdw.wtf.persistence.meta.StringValue;
import de.fhdw.wtf.persistence.meta.Transaction;
import de.fhdw.wtf.persistence.meta.UnidirectionalAssociation;
import de.fhdw.wtf.persistence.meta.UnidirectionalLink;
import de.fhdw.wtf.persistence.meta.UserObject;
import de.fhdw.wtf.persistence.meta.UserType;
import de.fhdw.wtf.persistence.utils.DBConnectionIntegerHandler;
import de.fhdw.wtf.persistence.utils.DBConnectionObjectHandler;
import de.fhdw.wtf.persistence.utils.DBConnectionStringHandler;
import de.fhdw.wtf.persistence.utils.DBConnectionUserObjectHandler;
import de.fhdw.wtf.persistence.utils.IntegerConstants;
import de.fhdw.wtf.persistence.utils.StringConstants;
import de.fhdw.wtf.persistence.utils.Tuple;

/**
 * A class to represent the implementation of the Object Facade Interface for the Oracle Database.
 * 
 */
public class OracleObjectFacadeImplementation implements ObjectFacade {
	
	private static final String CALL_SUFFIX = " end;";
	
	private static final String STORED_PROCEDURE_PREFIX = "begin ";
	
	private static final String STORED_FUNCTION_PREFIX = "begin ? := ";
	
	/**
	 * The Oracle Database Manager which is needed to call stored procedures in the database.
	 */
	private final OracleDatabaseManager database;
	
	/**
	 * The Type Manager, which has the Information about all model items.
	 */
	private final TypeManager typeManager;
	
	/**
	 * Constructor for a new Oracle Object Facade Implementation class.
	 * 
	 * @param database
	 *            An Oracle Database Manager, which is needed to access the Database.
	 * @param typeManager
	 *            A type Manager needed to determine type information.
	 */
	public OracleObjectFacadeImplementation(final OracleDatabaseManager database, final TypeManager typeManager) {
		super();
		this.database = database;
		this.typeManager = typeManager;
		IntegerValue.setObjectFacade(this);
		StringValue.setObjectFacade(this);
	}
	
	@Override
	public Collection<UserObject> find(final Association association, final String string, final Date date)
			throws PersistenceException {
		try (final CallableStatement call =
				this.database.getConnection().prepareCall(
						STORED_FUNCTION_PREFIX + this.database.getSchemaName() + ".objectfacade.findString(?,?,?);"
								+ CALL_SUFFIX)) {
			call.registerOutParameter(1, OracleTypes.CURSOR);
			call.setLong(IntegerConstants.TWO, association.getId());
			call.setString(IntegerConstants.THREE, string);
			call.setTimestamp(IntegerConstants.FOUR, new Timestamp(date.getTime()));
			call.execute();
			
			try (final ResultSet resultSet = ((OracleCallableStatement) call).getCursor(1)) {
				final Collection<UserObject> result = new HashSet<>();
				while (resultSet.next()) {
					result.add(UserObject.init(
							resultSet.getLong(1),
							this.typeManager.getTypeForId(resultSet.getLong(2))));
				}
				return result;
			}
		} catch (final SQLException e) {
			throw new OtherSQLException(e);
		}
	}
	
	@Override
	public Collection<UserObject> find(final Association association, final BigInteger integer, final Date date)
			throws PersistenceException {
		try (final CallableStatement call =
				this.database.getConnection().prepareCall(
						STORED_FUNCTION_PREFIX + this.database.getSchemaName() + ".objectfacade.findInteger(?,?,?);"
								+ CALL_SUFFIX)) {
			call.registerOutParameter(1, OracleTypes.CURSOR);
			call.setLong(2, association.getId());
			final BigDecimal wrapper = new BigDecimal(integer);
			call.setBigDecimal(IntegerConstants.THREE, wrapper);
			call.setTimestamp(IntegerConstants.FOUR, new Timestamp(date.getTime()));
			call.execute();
			
			try (final ResultSet resultSet = ((OracleCallableStatement) call).getCursor(1)) {
				final Collection<UserObject> result = new HashSet<>();
				while (resultSet.next()) {
					result.add(UserObject.init(
							resultSet.getLong(1),
							this.typeManager.getTypeForId(resultSet.getLong(2))));
				}
				return result;
			}
		} catch (final SQLException e) {
			throw new OtherSQLException(e);
		}
	}
	
	@Override
	public Collection<Tuple<UnidirectionalLink, Object>> get(final UserObject owner,
			final UnidirectionalAssociation association,
			final Date date) throws PersistenceException {
		try (final CallableStatement call =
				this.database.getConnection().prepareCall(
						STORED_FUNCTION_PREFIX + this.database.getSchemaName() + ".objectfacade.get(?,?,?);"
								+ CALL_SUFFIX)) {
			call.registerOutParameter(1, OracleTypes.CURSOR);
			call.setLong(2, owner.getId());
			call.setLong(IntegerConstants.THREE, association.getId());
			call.setTimestamp(IntegerConstants.FOUR, new Timestamp(date.getTime()));
			call.execute();
			
			try (final ResultSet resultSet = ((OracleCallableStatement) call).getCursor(1)) {
				final Collection<Tuple<UnidirectionalLink, Object>> result = new ArrayList<>();
				while (resultSet.next()) {
					final Object target =
							Object.checkForBaseTypes(
									resultSet.getLong(3),
									this.typeManager.getTypeForId(resultSet.getLong(4)));
					result.add(new Tuple<>(new UnidirectionalLink(resultSet.getLong(1), owner, target, this.typeManager
							.getUnidirectionalAssociationForId(resultSet.getLong(2))), target));
				}
				return result;
			}
		} catch (final SQLException e) {
			throw new OtherSQLException(e);
		}
	}
	
	private Collection<Object> getWithKeyHelper(final UserObject owner,
			final MapAssociation association,
			final DBConnectionObjectHandler keyHandler,
			final Date date) throws PersistenceException {
		try (final CallableStatement call =
				this.database.getConnection().prepareCall(
						STORED_FUNCTION_PREFIX + this.database.getSchemaName() + StringConstants.OBJECTFACADEGETMAP
								+ keyHandler.getObjectTypeString() + StringConstants.FOURQUESTIONMARKS + CALL_SUFFIX)) {
			call.registerOutParameter(1, OracleTypes.CURSOR);
			call.setLong(2, owner.getId());
			call.setLong(IntegerConstants.THREE, association.getId());
			keyHandler.handleCall(call, IntegerConstants.FOUR);
			call.setTimestamp(IntegerConstants.FIVE, new Timestamp(date.getTime()));
			call.execute();
			
			try (final ResultSet resultSet = ((OracleCallableStatement) call).getCursor(1)) {
				final Collection<Object> result = new ArrayList<>();
				while (resultSet.next()) {
					if (!result.isEmpty()) {
						throw new ToManyObjectsException();
					}
					final Object target =
							Object.checkForBaseTypes(
									resultSet.getLong(1),
									this.typeManager.getTypeForId(resultSet.getLong(2)));
					result.add(target);
				}
				if (result.isEmpty()) {
					throw new ObjectNotFoundException();
				}
				return result;
			}
		} catch (final SQLException e) {
			throw new OtherSQLException(e);
		}
	}
	
	@Override
	public Collection<Object> get(final UserObject owner,
			final MapAssociation association,
			final String key,
			final Date date) throws PersistenceException {
		return this.getWithKeyHelper(owner, association, new DBConnectionStringHandler(key), date);
	}
	
	@Override
	public Collection<Object> get(final UserObject owner,
			final MapAssociation association,
			final BigInteger key,
			final Date date) throws PersistenceException {
		return this.getWithKeyHelper(owner, association, new DBConnectionIntegerHandler(key), date);
	}
	
	@Override
	public Collection<Object> get(final UserObject owner,
			final MapAssociation association,
			final UserObject key,
			final Date date) throws PersistenceException {
		return this.getWithKeyHelper(owner, association, new DBConnectionUserObjectHandler(key), date);
	}
	
	@Override
	public Collection<Tuple<UnidirectionalLink, UserObject>> inverseGet(final UserObject target,
			final UnidirectionalAssociation association,
			final Date date) throws PersistenceException {
		try (final CallableStatement call =
				this.database.getConnection().prepareCall(
						STORED_FUNCTION_PREFIX + this.database.getSchemaName() + ".objectfacade.inverseGet(?,?,?);"
								+ CALL_SUFFIX)) {
			call.registerOutParameter(1, OracleTypes.CURSOR);
			call.setLong(2, target.getId());
			call.setLong(IntegerConstants.THREE, association.getId());
			call.setTimestamp(IntegerConstants.FOUR, new Timestamp(date.getTime()));
			call.execute();
			
			try (final ResultSet resultSet = ((OracleCallableStatement) call).getCursor(1)) {
				final Collection<Tuple<UnidirectionalLink, UserObject>> result = new ArrayList<>();
				while (resultSet.next()) {
					final UserObject owner =
							UserObject.init(resultSet.getLong(3), this.typeManager.getTypeForId(resultSet.getLong(4)));
					result.add(new Tuple<>(new UnidirectionalLink(resultSet.getLong(1), owner, target, this.typeManager
							.getUnidirectionalAssociationForId(resultSet.getLong(2))), owner));
				}
				return result;
			}
		} catch (final SQLException e) {
			throw new OtherSQLException(e);
		}
	}
	
	private Collection<Tuple<UserObject, Object>> inverseGetWithKeyHelper(final DBConnectionObjectHandler targetHandler,
			final MapAssociation association,
			final Date date) throws PersistenceException {
		try (final CallableStatement call =
				this.database.getConnection().prepareCall(
						STORED_FUNCTION_PREFIX + this.database.getSchemaName()
								+ StringConstants.OBJECTFACADEINVERSEGETMAP + targetHandler.getObjectTypeString()
								+ "(?,?,?);" + CALL_SUFFIX)) {
			call.registerOutParameter(1, OracleTypes.CURSOR);
			targetHandler.handleCall(call, 2);
			call.setLong(IntegerConstants.THREE, association.getId());
			call.setTimestamp(IntegerConstants.FOUR, new Timestamp(date.getTime()));
			call.execute();
			
			try (final ResultSet resultSet = ((OracleCallableStatement) call).getCursor(1)) {
				final Collection<Tuple<UserObject, Object>> owners = new ArrayList<>();
				while (resultSet.next()) {
					owners.add(new Tuple<>(UserObject.init(
							resultSet.getLong(1),
							this.typeManager.getTypeForId(resultSet.getLong(2))), Object.checkForBaseTypes(
							resultSet.getLong(IntegerConstants.THREE),
							this.typeManager.getTypeForId(resultSet.getLong(IntegerConstants.FOUR)))));
				}
				return owners;
			}
		} catch (final SQLException e) {
			throw new OtherSQLException(e);
		}
	}
	
	@Override
	public Collection<Tuple<UserObject, Object>> inverseGet(final UserObject target,
			final MapAssociation association,
			final Date date) throws PersistenceException {
		return this.inverseGetWithKeyHelper(new DBConnectionUserObjectHandler(target), association, date);
	}
	
	@Override
	public Collection<Tuple<UserObject, Object>> inverseGet(final String target,
			final MapAssociation association,
			final Date date) throws PersistenceException {
		return this.inverseGetWithKeyHelper(new DBConnectionStringHandler(target), association, date);
	}
	
	@Override
	public Collection<Tuple<UserObject, Object>> inverseGet(final BigInteger target,
			final MapAssociation association,
			final Date date) throws PersistenceException {
		return this.inverseGetWithKeyHelper(new DBConnectionIntegerHandler(target), association, date);
	}
	
	@Override
	public Collection<UserObject> find(final Association association, final String string, final Transaction transaction)
			throws PersistenceException {
		this.checkTransaction(transaction);
		try (final CallableStatement call =
				this.database.getConnection().prepareCall(
						STORED_FUNCTION_PREFIX + this.database.getSchemaName()
								+ ".objectfacade.findStringTransaction(?,?,?);" + CALL_SUFFIX)) {
			call.registerOutParameter(1, OracleTypes.CURSOR);
			call.setLong(2, association.getId());
			call.setString(IntegerConstants.THREE, string);
			call.setLong(IntegerConstants.FOUR, transaction.getId());
			call.execute();
			
			try (final ResultSet resultSet = ((OracleCallableStatement) call).getCursor(1)) {
				final Collection<UserObject> result = new HashSet<>();
				while (resultSet.next()) {
					result.add(UserObject.init(
							resultSet.getLong(1),
							this.typeManager.getTypeForId(resultSet.getLong(2))));
				}
				return result;
			}
		} catch (final SQLException e) {
			throw new OtherSQLException(e);
		}
	}
	
	/**
	 * This method checks if a given transaction is really an open transaction and throws an exception if not.
	 * 
	 * @param transaction
	 *            An Transaction.
	 * @throws PersistenceException
	 *             This Exception occurs if there are any communicating issues.
	 * @throws NoValidTransactionException
	 *             This Exception is the Transaction is not an open transaction.
	 */
	private void checkTransaction(final Transaction transaction) throws PersistenceException,
			NoValidTransactionException {
		if (!this.isOpenTransaction(transaction)) {
			throw new NoValidTransactionException();
		}
	}
	
	@Override
	public Collection<UserObject> find(final Association association,
			final BigInteger integer,
			final Transaction transaction) throws PersistenceException {
		this.checkTransaction(transaction);
		try (final CallableStatement call =
				this.database.getConnection().prepareCall(
						STORED_FUNCTION_PREFIX + this.database.getSchemaName()
								+ ".objectfacade.findIntegerTransaction(?,?,?);" + CALL_SUFFIX)) {
			call.registerOutParameter(1, OracleTypes.CURSOR);
			call.setLong(2, association.getId());
			final BigDecimal wrapper = new BigDecimal(integer);
			call.setBigDecimal(IntegerConstants.THREE, wrapper);
			call.setLong(IntegerConstants.FOUR, transaction.getId());
			call.execute();
			
			try (final ResultSet resultSet = ((OracleCallableStatement) call).getCursor(1)) {
				final Collection<UserObject> result = new HashSet<>();
				while (resultSet.next()) {
					result.add(UserObject.init(
							resultSet.getLong(1),
							this.typeManager.getTypeForId(resultSet.getLong(2))));
				}
				return result;
			}
		} catch (final SQLException e) {
			throw new OtherSQLException(e);
		}
	}
	
	@Override
	public Collection<Tuple<UnidirectionalLink, Object>> get(final UserObject owner,
			final UnidirectionalAssociation association,
			final Transaction transaction) throws PersistenceException {
		this.checkTransaction(transaction);
		try (final CallableStatement call =
				this.database.getConnection().prepareCall(
						STORED_FUNCTION_PREFIX + this.database.getSchemaName() + ".objectfacade.getTransaction(?,?,?);"
								+ CALL_SUFFIX)) {
			call.registerOutParameter(1, OracleTypes.CURSOR);
			call.setLong(2, owner.getId());
			call.setLong(IntegerConstants.THREE, association.getId());
			call.setLong(IntegerConstants.FOUR, transaction.getId());
			call.execute();
			
			try (final ResultSet resultSet = ((OracleCallableStatement) call).getCursor(1)) {
				final Collection<Tuple<UnidirectionalLink, Object>> result = new ArrayList<>();
				while (resultSet.next()) {
					final Object target =
							Object.checkForBaseTypes(
									resultSet.getLong(3),
									this.typeManager.getTypeForId(resultSet.getLong(4)));
					result.add(new Tuple<>(new UnidirectionalLink(resultSet.getLong(1), owner, target, this.typeManager
							.getUnidirectionalAssociationForId(resultSet.getLong(2))), target));
				}
				return result;
			}
		} catch (final SQLException e) {
			throw new OtherSQLException(e);
		}
	}
	
	private Collection<Tuple<MapLink, Object>> getWithKeyHelper(final UserObject owner,
			final MapAssociation a,
			final DBConnectionObjectHandler keyHandler,
			final Transaction transaction) throws PersistenceException {
		this.checkTransaction(transaction);
		try (final CallableStatement call =
				this.database.getConnection().prepareCall(
						STORED_FUNCTION_PREFIX + this.database.getSchemaName() + ".objectfacade.getMap"
								+ keyHandler.getObjectTypeString() + "Transaction(?,?,?,?);" + CALL_SUFFIX)) {
			call.registerOutParameter(1, OracleTypes.CURSOR);
			call.setLong(2, owner.getId());
			call.setLong(IntegerConstants.THREE, a.getId());
			keyHandler.handleCall(call, IntegerConstants.FOUR);
			call.setLong(IntegerConstants.FIVE, transaction.getId());
			call.execute();
			
			try (final ResultSet resultSet = ((OracleCallableStatement) call).getCursor(1)) {
				final Collection<Tuple<MapLink, Object>> result = new ArrayList<>();
				while (resultSet.next()) {
					if (!result.isEmpty()) {
						throw new ToManyObjectsException();
					}
					final Object target =
							Object.checkForBaseTypes(
									resultSet.getLong(1),
									this.typeManager.getTypeForId(resultSet.getLong(2)));
					final MapLink link =
							new MapLink(resultSet.getLong(IntegerConstants.FIVE), owner, target,
									keyHandler.getObject(), a);
					result.add(new Tuple<>(link, target));
				}
				if (result.isEmpty()) {
					throw new ObjectNotFoundException();
				}
				return result;
			}
		} catch (final SQLException e) {
			throw new OtherSQLException(e);
		}
	}
	
	@Override
	public Collection<Tuple<MapLink, Object>> get(final UserObject owner,
			final MapAssociation association,
			final String key,
			final Transaction transaction) throws PersistenceException {
		return this.getWithKeyHelper(owner, association, new DBConnectionStringHandler(key), transaction);
	}
	
	@Override
	public Collection<Tuple<MapLink, Object>> get(final UserObject owner,
			final MapAssociation association,
			final BigInteger key,
			final Transaction transaction) throws PersistenceException {
		return this.getWithKeyHelper(owner, association, new DBConnectionIntegerHandler(key), transaction);
	}
	
	@Override
	public Collection<Tuple<MapLink, Object>> get(final UserObject owner,
			final MapAssociation association,
			final UserObject key,
			final Transaction transaction) throws PersistenceException {
		return this.getWithKeyHelper(owner, association, new DBConnectionUserObjectHandler(key), transaction);
	}
	
	@Override
	public Collection<Tuple<UnidirectionalLink, UserObject>> inverseGet(final UserObject target,
			final UnidirectionalAssociation association,
			final Transaction transaction) throws PersistenceException {
		this.checkTransaction(transaction);
		try (final CallableStatement call =
				this.database.getConnection().prepareCall(
						STORED_FUNCTION_PREFIX + this.database.getSchemaName()
								+ ".objectfacade.inverseGetTransaction(?,?,?);" + CALL_SUFFIX)) {
			call.registerOutParameter(1, OracleTypes.CURSOR);
			call.setLong(2, target.getId());
			call.setLong(IntegerConstants.THREE, association.getId());
			call.setLong(IntegerConstants.FOUR, transaction.getId());
			call.execute();
			
			try (final ResultSet resultSet = ((OracleCallableStatement) call).getCursor(1)) {
				final Collection<Tuple<UnidirectionalLink, UserObject>> result = new ArrayList<>();
				while (resultSet.next()) {
					final UserObject owner =
							UserObject.init(resultSet.getLong(3), this.typeManager.getTypeForId(resultSet.getLong(4)));
					result.add(new Tuple<>(new UnidirectionalLink(resultSet.getLong(1), owner, target, this.typeManager
							.getUnidirectionalAssociationForId(resultSet.getLong(2))), owner));
				}
				return result;
			}
		} catch (final SQLException e) {
			throw new OtherSQLException(e);
		}
	}
	
	private Collection<Tuple<UserObject, Object>> inverseGetWithKeyHelper(final DBConnectionObjectHandler targetHandler,
			final MapAssociation association,
			final Transaction transaction) throws PersistenceException {
		this.checkTransaction(transaction);
		try (final CallableStatement call =
				this.database.getConnection().prepareCall(
						STORED_FUNCTION_PREFIX + this.database.getSchemaName() + ".objectfacade.inverseGetMap"
								+ targetHandler.getObjectTypeString() + "Transaction(?,?,?);" + CALL_SUFFIX)) {
			call.registerOutParameter(1, OracleTypes.CURSOR);
			targetHandler.handleCall(call, 2);
			call.setLong(IntegerConstants.THREE, association.getId());
			call.setLong(IntegerConstants.FOUR, transaction.getId());
			call.execute();
			
			try (final ResultSet resultSet = ((OracleCallableStatement) call).getCursor(1)) {
				final Collection<Tuple<UserObject, Object>> owners = new ArrayList<>();
				while (resultSet.next()) {
					owners.add(new Tuple<>(UserObject.init(
							resultSet.getLong(1),
							this.typeManager.getTypeForId(resultSet.getLong(2))), Object.checkForBaseTypes(
							resultSet.getLong(IntegerConstants.THREE),
							this.typeManager.getTypeForId(resultSet.getLong(IntegerConstants.FOUR)))));
				}
				return owners;
			}
		} catch (final SQLException e) {
			throw new OtherSQLException(e);
		}
	}
	
	@Override
	public Collection<Tuple<UserObject, Object>> inverseGet(final UserObject target,
			final MapAssociation association,
			final Transaction transaction) throws PersistenceException {
		return this.inverseGetWithKeyHelper(new DBConnectionUserObjectHandler(target), association, transaction);
	}
	
	@Override
	public Collection<Tuple<UserObject, Object>> inverseGet(final String target,
			final MapAssociation association,
			final Transaction transaction) throws PersistenceException {
		return this.inverseGetWithKeyHelper(new DBConnectionStringHandler(target), association, transaction);
	}
	
	@Override
	public Collection<Tuple<UserObject, Object>> inverseGet(final BigInteger target,
			final MapAssociation association,
			final Transaction transaction) throws PersistenceException {
		return this.inverseGetWithKeyHelper(new DBConnectionIntegerHandler(target), association, transaction);
	}
	
	@Override
	public boolean isInConflict(final Transaction transaction1, final Transaction transaction2)
			throws PersistenceException {
		this.checkTransaction(transaction1);
		this.checkTransaction(transaction2);
		try (final CallableStatement call =
				this.database.getConnection().prepareCall(
						STORED_FUNCTION_PREFIX + this.database.getSchemaName()
								+ ".objectfacade.getConflictingTransactions(?);" + CALL_SUFFIX)) {
			call.registerOutParameter(1, OracleTypes.CURSOR);
			call.setLong(2, transaction1.getId());
			call.execute();
			
			try (final ResultSet resultSet = ((OracleCallableStatement) call).getCursor(1)) {
				while (resultSet.next()) {
					if (transaction2.getId() == resultSet.getLong(1)) {
						return true;
					}
				}
				return false;
			}
		} catch (final SQLException e) {
			throw new OtherSQLException(e);
		}
	}
	
	@Override
	public boolean isOpenTransaction(final Transaction transaction) throws PersistenceException {
		try (final CallableStatement call =
				this.database.getConnection().prepareCall(
						STORED_FUNCTION_PREFIX + this.database.getSchemaName() + ".objectfacade.isTransactionOpen(?);"
								+ CALL_SUFFIX)) {
			call.registerOutParameter(1, OracleTypes.NUMBER);
			call.setLong(2, transaction.getId());
			call.execute();
			return call.getInt(1) == 1 ? true : false;
		} catch (final SQLException e) {
			throw new OtherSQLException(e);
		}
	}
	
	@Override
	public UserObject create(final UserType type, final Transaction transaction) throws PersistenceException {
		this.checkTransaction(transaction);
		try (final CallableStatement call =
				this.database.getConnection().prepareCall(
						STORED_FUNCTION_PREFIX + this.database.getSchemaName() + ".objectfacade.createUserObject(?,?);"
								+ CALL_SUFFIX)) {
			call.registerOutParameter(1, OracleTypes.NUMBER);
			call.setLong(2, type.getId());
			call.setLong(IntegerConstants.THREE, transaction.getId());
			call.execute();
			return UserObject.init(call.getLong(1), type);
		} catch (final SQLException e) {
			if (e.getErrorCode() == NotInstantiatableException.ERRORCODE) {
				throw new NotInstantiatableException(e);
			}
			e.printStackTrace();
			throw new OtherSQLException(e);
		}
	}
	
	@Override
	public void delete(final UserObject object, final Transaction transaction) throws PersistenceException {
		this.checkTransaction(transaction);
		try (final CallableStatement call =
				this.database.getConnection().prepareCall(
						STORED_PROCEDURE_PREFIX + this.database.getSchemaName()
								+ ".objectfacade.deleteUserObject(?,?);" + CALL_SUFFIX)) {
			call.setLong(1, object.getId());
			call.setLong(2, transaction.getId());
			call.execute();
		} catch (final SQLException e) {
			if (e.getErrorCode() == InvalidLinkException.ERRORCODE) {
				throw new InvalidLinkException(e);
			}
			throw new OtherSQLException(e);
		}
	}
	
	private UnidirectionalLink setHelper(final UserObject owner,
			final UnidirectionalAssociation association,
			final DBConnectionObjectHandler targetHandler,
			final Transaction transaction) throws PersistenceException {
		this.checkTransaction(transaction);
		try (final CallableStatement call =
				this.database.getConnection().prepareCall(
						STORED_FUNCTION_PREFIX + this.database.getSchemaName() + ".objectfacade.set"
								+ targetHandler.getObjectTypeString() + "(?,?,?,?);" + CALL_SUFFIX)) {
			call.registerOutParameter(1, OracleTypes.NUMBER);
			call.setLong(2, owner.getId());
			call.setLong(IntegerConstants.THREE, association.getId());
			targetHandler.handleCall(call, IntegerConstants.FOUR);
			call.setLong(IntegerConstants.FIVE, transaction.getId());
			call.execute();
			return new UnidirectionalLink(call.getLong(1), owner, targetHandler.getObject(), association);
		} catch (final SQLException e) {
			if (e.getErrorCode() == InvalidLinkException.ERRORCODE) {
				throw new InvalidLinkException(e);
			}
			if (e.getErrorCode() == NotInScopeException.ERRORCODE) {
				throw new NotInScopeException(e);
			}
			throw new OtherSQLException(e);
		}
	}
	
	@Override
	public UnidirectionalLink set(final UserObject owner,
			final UnidirectionalAssociation association,
			final UserObject target,
			final Transaction transaction) throws PersistenceException {
		return this.setHelper(owner, association, new DBConnectionUserObjectHandler(target), transaction);
	}
	
	@Override
	public UnidirectionalLink set(final UserObject owner,
			final UnidirectionalAssociation association,
			final BigInteger target,
			final Transaction transaction) throws PersistenceException {
		return this.setHelper(owner, association, new DBConnectionIntegerHandler(target), transaction);
	}
	
	@Override
	public UnidirectionalLink set(final UserObject owner,
			final UnidirectionalAssociation association,
			final String target,
			final Transaction transaction) throws PersistenceException {
		return this.setHelper(owner, association, new DBConnectionStringHandler(target), transaction);
	}
	
	private MapLink putHelper(final UserObject owner,
			final MapAssociation association,
			final DBConnectionObjectHandler targetHandler,
			final DBConnectionObjectHandler keyHandler,
			final Transaction transaction) throws PersistenceException {
		try (final CallableStatement call =
				this.database.getConnection().prepareCall(
						STORED_FUNCTION_PREFIX + this.database.getSchemaName() + ".objectfacade.put"
								+ targetHandler.getObjectTypeString() + keyHandler.getObjectTypeString()
								+ "(?,?,?,?,?);" + CALL_SUFFIX)) {
			call.registerOutParameter(1, OracleTypes.NUMBER);
			call.setLong(2, owner.getId());
			call.setLong(IntegerConstants.THREE, association.getId());
			targetHandler.handleCall(call, IntegerConstants.FOUR);
			keyHandler.handleCall(call, IntegerConstants.FIVE);
			call.setLong(IntegerConstants.SIX, transaction.getId());
			call.execute();
			return new MapLink(call.getLong(1), owner, targetHandler.getObject(), keyHandler.getObject(), association);
		} catch (final SQLException e) {
			if (e.getErrorCode() == InvalidLinkException.ERRORCODE) {
				throw new InvalidLinkException(e);
			}
			if (e.getErrorCode() == NotInScopeException.ERRORCODE) {
				throw new NotInScopeException(e);
			}
			if (e.getErrorCode() == DuplicateEntryException.ERRORCODE) {
				throw new DuplicateEntryException(e);
			}
			throw new OtherSQLException(e);
		}
	}
	
	@Override
	public MapLink put(final UserObject owner,
			final MapAssociation association,
			final UserObject target,
			final String key,
			final Transaction transaction) throws PersistenceException {
		return this.putHelper(
				owner,
				association,
				new DBConnectionUserObjectHandler(target),
				new DBConnectionStringHandler(key),
				transaction);
	}
	
	@Override
	public MapLink put(final UserObject owner,
			final MapAssociation association,
			final BigInteger target,
			final String key,
			final Transaction transaction) throws PersistenceException {
		return this.putHelper(
				owner,
				association,
				new DBConnectionIntegerHandler(target),
				new DBConnectionStringHandler(key),
				transaction);
	}
	
	@Override
	public MapLink put(final UserObject owner,
			final MapAssociation association,
			final String target,
			final String key,
			final Transaction transaction) throws PersistenceException {
		return this.putHelper(owner, association, new DBConnectionStringHandler(target), new DBConnectionStringHandler(
				key), transaction);
	}
	
	@Override
	public MapLink put(final UserObject owner,
			final MapAssociation association,
			final UserObject target,
			final BigInteger key,
			final Transaction transaction) throws PersistenceException {
		return this.putHelper(
				owner,
				association,
				new DBConnectionUserObjectHandler(target),
				new DBConnectionIntegerHandler(key),
				transaction);
	}
	
	@Override
	public MapLink put(final UserObject owner,
			final MapAssociation association,
			final BigInteger target,
			final BigInteger key,
			final Transaction transaction) throws PersistenceException {
		return this.putHelper(
				owner,
				association,
				new DBConnectionIntegerHandler(target),
				new DBConnectionIntegerHandler(key),
				transaction);
	}
	
	@Override
	public MapLink put(final UserObject owner,
			final MapAssociation association,
			final String target,
			final BigInteger key,
			final Transaction transaction) throws PersistenceException {
		return this.putHelper(
				owner,
				association,
				new DBConnectionStringHandler(target),
				new DBConnectionIntegerHandler(key),
				transaction);
	}
	
	@Override
	public MapLink put(final UserObject owner,
			final MapAssociation association,
			final UserObject target,
			final UserObject key,
			final Transaction transaction) throws PersistenceException {
		return this.putHelper(
				owner,
				association,
				new DBConnectionUserObjectHandler(target),
				new DBConnectionUserObjectHandler(key),
				transaction);
	}
	
	@Override
	public MapLink put(final UserObject owner,
			final MapAssociation association,
			final BigInteger target,
			final UserObject key,
			final Transaction transaction) throws PersistenceException {
		return this.putHelper(
				owner,
				association,
				new DBConnectionIntegerHandler(target),
				new DBConnectionUserObjectHandler(key),
				transaction);
	}
	
	@Override
	public MapLink put(final UserObject owner,
			final MapAssociation association,
			final String target,
			final UserObject key,
			final Transaction transaction) throws PersistenceException {
		return this.putHelper(
				owner,
				association,
				new DBConnectionStringHandler(target),
				new DBConnectionUserObjectHandler(key),
				transaction);
	}
	
	@Override
	public void unset(final Link link, final Transaction transaction) throws PersistenceException {
		this.checkTransaction(transaction);
		try (final CallableStatement call =
				this.database.getConnection().prepareCall(
						STORED_PROCEDURE_PREFIX + this.database.getSchemaName() + ".objectfacade.unset(?,?);"
								+ CALL_SUFFIX)) {
			call.setLong(1, link.getId());
			call.setLong(2, transaction.getId());
			call.execute();
		} catch (final SQLException e) {
			throw new OtherSQLException(e);
		}
	}
	
	@Override
	public void commit(final Transaction transaction) throws PersistenceException {
		this.checkTransaction(transaction);
		try (final CallableStatement call =
				this.database.getConnection().prepareCall(
						STORED_PROCEDURE_PREFIX + this.database.getSchemaName() + ".objectfacade.commitTransaction(?);"
								+ CALL_SUFFIX)) {
			call.setLong(1, transaction.getId());
			call.execute();
		} catch (final SQLException e) {
			throw new OtherSQLException(e);
		}
	}
	
	@Override
	public void rollback(final Transaction transaction) throws PersistenceException {
		this.checkTransaction(transaction);
		try (final CallableStatement call =
				this.database.getConnection().prepareCall(
						STORED_PROCEDURE_PREFIX + this.database.getSchemaName()
								+ ".objectfacade.rollbackTransaction(?);" + CALL_SUFFIX)) {
			call.setLong(1, transaction.getId());
			call.execute();
		} catch (final SQLException e) {
			throw new OtherSQLException(e);
		}
	}
	
	@Override
	public void savePoint(final Transaction transaction) throws PersistenceException {
		this.checkTransaction(transaction);
		try (final CallableStatement call =
				this.database.getConnection().prepareCall(
						STORED_PROCEDURE_PREFIX + this.database.getSchemaName() + ".objectfacade.makeSavePoint(?);"
								+ CALL_SUFFIX)) {
			call.setLong(1, transaction.getId());
			call.execute();
		} catch (final SQLException e) {
			throw new OtherSQLException(e);
		}
	}
	
	@Override
	public void rollbackToSavePoint(final Transaction transaction) throws PersistenceException {
		this.checkTransaction(transaction);
		try (final CallableStatement call =
				this.database.getConnection().prepareCall(
						STORED_PROCEDURE_PREFIX + this.database.getSchemaName()
								+ ".objectfacade.rollbackToSavePoint(?);" + CALL_SUFFIX)) {
			call.setLong(1, transaction.getId());
			call.execute();
		} catch (final SQLException e) {
			throw new OtherSQLException(e);
		}
	}
	
	@Override
	public BigInteger getIntForId(final long id) throws PersistenceException {
		try (final CallableStatement call =
				this.database.getConnection().prepareCall(
						STORED_FUNCTION_PREFIX + this.database.getSchemaName() + ".objectfacade.getInteger(?);"
								+ CALL_SUFFIX)) {
			call.registerOutParameter(1, OracleTypes.NUMBER);
			call.setLong(2, id);
			call.execute();
			return BigInteger.valueOf(call.getLong(1));
		} catch (final SQLException e) {
			if (e.getErrorCode() == BaseTypeNotFoundException.ERRORCODE) {
				throw new BaseTypeNotFoundException();
			}
			throw new OtherSQLException(e);
		}
	}
	
	@Override
	public String getStringForId(final long id) throws PersistenceException {
		try (final CallableStatement call =
				this.database.getConnection().prepareCall(
						STORED_FUNCTION_PREFIX + this.database.getSchemaName() + ".objectfacade.getString(?);"
								+ CALL_SUFFIX)) {
			call.registerOutParameter(1, OracleTypes.VARCHAR);
			call.setLong(2, id);
			call.execute();
			return call.getString(1);
		} catch (final SQLException e) {
			if (e.getErrorCode() == BaseTypeNotFoundException.ERRORCODE) {
				throw new BaseTypeNotFoundException();
			}
			throw new OtherSQLException(e);
		}
	}
	
	@Override
	public long getIdForString(final String string) throws PersistenceException {
		try (final CallableStatement call =
				this.database.getConnection().prepareCall(
						STORED_FUNCTION_PREFIX + this.database.getSchemaName() + ".objectfacade.getStringObject(?);"
								+ CALL_SUFFIX)) {
			call.registerOutParameter(1, OracleTypes.NUMBER);
			call.setString(2, string);
			call.execute();
			return call.getLong(1);
		} catch (final SQLException e) {
			throw new OtherSQLException(e);
		}
	}
	
	@Override
	public long getIdForInteger(final BigInteger integer) throws PersistenceException {
		try (final CallableStatement call =
				this.database.getConnection().prepareCall(
						STORED_FUNCTION_PREFIX + this.database.getSchemaName() + ".objectfacade.getIntegerObject(?);"
								+ CALL_SUFFIX)) {
			call.registerOutParameter(1, OracleTypes.NUMBER);
			final BigDecimal wrapper = new BigDecimal(integer);
			call.setBigDecimal(2, wrapper);
			call.execute();
			return call.getLong(1);
		} catch (final SQLException e) {
			throw new OtherSQLException(e);
		}
	}
	
	@Override
	public Transaction provideAdhocTransaction() throws PersistenceException {
		try (final CallableStatement call =
				this.database.getConnection().prepareCall(
						STORED_FUNCTION_PREFIX + this.database.getSchemaName()
								+ ".objectfacade.provideAdhocTransaction;" + CALL_SUFFIX)) {
			call.registerOutParameter(1, OracleTypes.NUMBER);
			call.execute();
			return new AdhocTransaction(call.getLong(1));
		} catch (final SQLException e) {
			throw new OtherSQLException(e);
		}
	}
	
	@Override
	public TypeManager getTypeManager() {
		return this.typeManager;
	}
	
	@Override
	public void clear() throws PersistenceException {
		try (final CallableStatement call =
				this.database.getConnection().prepareCall(
						STORED_PROCEDURE_PREFIX + this.database.getSchemaName() + ".objectfacade.clear;" + CALL_SUFFIX)) {
			call.execute();
		} catch (final SQLException e) {
			e.printStackTrace();
			throw new OtherSQLException(e);
		}
	}
	
	@Override
	public Collection<UserObject> findAllObjects(final UserType type, final Transaction transaction)
			throws PersistenceException {
		final Collection<UserObject> result = new HashSet<>();
		try (final CallableStatement call =
				this.database.getConnection().prepareCall(
						STORED_FUNCTION_PREFIX + this.database.getSchemaName()
								+ ".objectfacade.findUserObjectByType(?,?);" + CALL_SUFFIX)) {
			call.registerOutParameter(1, OracleTypes.CURSOR);
			call.setLong(2, type.getId());
			call.setLong(IntegerConstants.THREE, transaction.getId());
			call.execute();
			try (final ResultSet resultSet = ((OracleCallableStatement) call).getCursor(1)) {
				while (resultSet.next()) {
					result.add(UserObject.init(
							resultSet.getLong(1),
							this.typeManager.getTypeForId(resultSet.getLong(2))));
				}
			}
		} catch (final SQLException e) {
			throw new OtherSQLException(e);
		}
		return result;
	}
	
	@Override
	public UserObject checkUserObjectOut(final long id, final Transaction transaction) throws PersistenceException {
		try (final CallableStatement call =
				this.database.getConnection().prepareCall(
						STORED_FUNCTION_PREFIX + this.database.getSchemaName()
								+ ".objectfacade.checkUserObjectOut(?,?);" + CALL_SUFFIX)) {
			call.registerOutParameter(1, OracleTypes.CURSOR);
			call.setLong(2, id);
			call.setLong(IntegerConstants.THREE, transaction.getId());
			call.execute();
			try (final ResultSet resultSet = ((OracleCallableStatement) call).getCursor(1)) {
				if (resultSet.next()) {
					return UserObject.init(resultSet.getLong(1), this.typeManager.getTypeForId(resultSet.getLong(2)));
				} else {
					throw new ObjectNotFoundException();
				}
			}
		} catch (final SQLException e) {
			throw new OtherSQLException(e);
		}
	}
}
