package de.fhdw.wtf.persistence.facade;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.sql.CallableStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Iterator;
import java.util.Properties;

import oracle.jdbc.OracleCallableStatement;
import oracle.jdbc.OracleTypes;
import de.fhdw.wtf.persistence.exception.IDContractViolationException;
import de.fhdw.wtf.persistence.exception.IDNotFoundForNameException;
import de.fhdw.wtf.persistence.exception.IDPersistenceFilesNotFound;
import de.fhdw.wtf.persistence.exception.OtherSQLException;
import de.fhdw.wtf.persistence.exception.PersistenceException;
import de.fhdw.wtf.persistence.meta.IntegerType;
import de.fhdw.wtf.persistence.meta.StringType;

/**
 * TODO generate the greatest id for associations and types dynamically from a set of the singleton base type classes.
 * 
 * Description: - This class manages the id distribution for types and associations. - If a new type or association is
 * created, the id for this new item should be pulled from this manager. - if you want to know the id for a model item
 * (association or type) ask this manager. - This class is also able to save the information about the current relation
 * (name -> id) to a file. See persistIDRelationsToFile. This should be called everytime you want to close the process
 * which has the idmanager instance. - On creation of this manager the (name -> id) relation will be loaded from a file
 * if it exists. Otherwise the database will be requested for the ids.
 */
public final class IDManager {
	/**
	 * The Id which was the highest given ID for types during the initialization, it must match the contract. Ids must
	 * be given in successive manner therefore a value of zero means there are no types initialized. This value is also
	 * contracted by the pl sql script for the classfacade. See initialize operation.
	 */
	private static final long MAX_BASE_TYPES = 2; // 2 because of 2 base types
													// {String, Integer}
	
	/**
	 * Because after the creation of the Base Types there must not be any association, the contracted value is zero.
	 */
	private static final long MAX_CONTRACT_ASSOCIATION_ID = 0;
	
	/** Specifies the filename of the file for the type name -> id relation. */
	public static final String TYPE_IDS_FILENAME = "types.properties";
	
	/**
	 * Specifies the filename of the file for the association name -> id relation.
	 */
	public static final String ASSOCIATION_IDS_FILENAME = "associations.properties";
	
	private static IDManager instance = null;
	private final Properties typeNameIdRelation;
	private final Properties associationNameIdRelation;
	private long currentlyUsedTypeMaxId;
	private long currentlyUsedAssociationMaxId;
	
	private IDManager() {
		this.typeNameIdRelation = new Properties();
		this.associationNameIdRelation = new Properties();
		this.currentlyUsedAssociationMaxId = getMaxAssociationContractID();
		this.currentlyUsedTypeMaxId = getMaxBaseTypeID();
	}
	
	/**
	 * This operation searches in the property the highest type id and sets the member currentlyUsedTypeMaxId to it.
	 * This is necessary to generate the next id unused it properly after the initialization of this class.
	 */
	private void recalibrateMaxTypeID() {
		final Iterator<String> it = this.typeNameIdRelation.stringPropertyNames().iterator();
		while (it.hasNext()) {
			final String currentKey = it.next();
			final Long currentId = Long.parseLong(this.typeNameIdRelation.getProperty(currentKey));
			if (this.currentlyUsedTypeMaxId < currentId) {
				this.currentlyUsedTypeMaxId = currentId;
			}
		}
		
	}
	
	/**
	 * This operation searches for the highest association id and sets the member currentlyUsedAssociationMaxId to it.
	 * This is necessary to generate the next id unused it properly after the initialization of this class.
	 */
	private void recalibrateMaxAssociationID() {
		final Iterator<String> it = this.associationNameIdRelation.stringPropertyNames().iterator();
		while (it.hasNext()) {
			final String currentKey = it.next();
			final Long currentId = Long.parseLong(this.associationNameIdRelation.getProperty(currentKey));
			if (this.currentlyUsedAssociationMaxId < currentId) {
				this.currentlyUsedAssociationMaxId = currentId;
			}
		}
	}
	
	/**
	 * Checks if the files with the id informations are available and initializes the internal data structures with
	 * these values.
	 * 
	 * @param typeIdsFilename
	 *            the filename to the file contains the typeIds
	 * @param associationIdsFilename
	 *            the filename to the file contains the assoctiationIds
	 * @throws IOException
	 * @throws FileNotFoundException
	 */
	public void initializeRelationsFromFile(final String typeIdsFilename, final String associationIdsFilename)
			throws FileNotFoundException, IOException {
		final File fileType = new File(typeIdsFilename);
		final File fileAssociation = new File(associationIdsFilename);
		
		if (fileType.exists() && !fileType.isDirectory() && fileAssociation.exists() && !fileAssociation.isDirectory()) {
			try (final FileInputStream fileTypeInputStream = new FileInputStream(fileType);
					final FileInputStream fileAssociationInputStream = new FileInputStream(fileAssociation)) {
				this.typeNameIdRelation.load(fileTypeInputStream);
				this.associationNameIdRelation.load(fileAssociationInputStream);
			}
		} else {
			throw new IDPersistenceFilesNotFound();
		}
		this.typeNameIdRelation.setProperty(StringType.STRING_NAME, Long.toString(StringType.String_ID));
		this.typeNameIdRelation.setProperty(IntegerType.INTEGER_NAME, Long.toString(IntegerType.Integer_ID));
		this.recalibrateMaxTypeID();
		this.recalibrateMaxAssociationID();
		
	}
	
	/**
	 * Initializes the name->id relations from the database. TODO create some operations in the classfacade which
	 * provide the types and associations directly from the database. This is not part of the id-manager.
	 * 
	 * @throws PersistenceException
	 * @throws SQLException
	 */
	public void initializeRelationsFromDatabase() throws PersistenceException, SQLException {
		final OracleDatabaseManager database = OracleDatabaseManager.getInstance();
		try {
			database.getConnection();
		} catch (final PersistenceException e) {
			database.connect();
		}
		
		try (final CallableStatement call =
				database.getConnection().prepareCall(
						"begin ?:= " + database.getSchemaName() + ".classfacade.getAllTypes; end;")) {
			call.registerOutParameter(1, OracleTypes.CURSOR);
			call.execute();
			
			try (final ResultSet result = ((OracleCallableStatement) call).getCursor(1)) {
				while (result.next()) {
					this.typeNameIdRelation.setProperty(result.getString(2), Long.toString(result.getLong(1)));
				}
			}
		} catch (final SQLException e) {
			throw new OtherSQLException(e);
		}
		
		try (final CallableStatement call =
				database.getConnection().prepareCall(
						"begin ?:= " + database.getSchemaName() + ".classfacade.getAllUnidirAssociations; end;")) {
			call.registerOutParameter(1, OracleTypes.CURSOR);
			call.execute();
			
			try (final ResultSet result = ((OracleCallableStatement) call).getCursor(1)) {
				while (result.next()) {
					this.associationNameIdRelation.setProperty(result.getString(2), Long.toString(result.getLong(1)));
				}
			}
		} catch (final SQLException e) {
			throw new OtherSQLException(e);
		}
		
		try (final CallableStatement call =
				database.getConnection().prepareCall(
						"begin ?:= " + database.getSchemaName() + ".classfacade.getAllMapAssociations; end;")) {
			call.registerOutParameter(1, OracleTypes.CURSOR);
			call.execute();
			
			try (final ResultSet result = ((OracleCallableStatement) call).getCursor(1)) {
				while (result.next()) {
					this.associationNameIdRelation.setProperty(result.getString(2), Long.toString(result.getLong(1)));
				}
			}
		} catch (final SQLException e) {
			throw new OtherSQLException(e);
		}
		
		this.typeNameIdRelation.setProperty(StringType.STRING_NAME, Long.toString(StringType.String_ID));
		this.typeNameIdRelation.setProperty(IntegerType.INTEGER_NAME, Long.toString(IntegerType.Integer_ID));
		this.recalibrateMaxTypeID();
		this.recalibrateMaxAssociationID();
	}
	
	/**
	 * returns the Instance of the IDManager.
	 * 
	 * @return the instance
	 */
	public static synchronized IDManager instance() {
		if (instance == null) {
			instance = new IDManager();
		}
		return instance;
	}
	
	/**
	 * Warning: Use it for Test-Cases only! This operations forces a recreation of the singleton for the next time and
	 * frees the old one for garbage collection.
	 */
	public void clearInformation() {
		this.typeNameIdRelation.clear();
		this.associationNameIdRelation.clear();
		this.currentlyUsedAssociationMaxId = getMaxAssociationContractID();
		this.currentlyUsedTypeMaxId = getMaxBaseTypeID();
		
	}
	
	/**
	 * 
	 * @return the highest id for base types.
	 */
	public static long getMaxBaseTypeID() {
		return MAX_BASE_TYPES;
	}
	
	/**
	 * @return the highest association id which will be initialized by the pl-sql script at initialization. The highest
	 *         id is contracted.
	 */
	public static long getMaxAssociationContractID() {
		return MAX_CONTRACT_ASSOCIATION_ID;
	}
	
	/**
	 * This operation persists all (name -> id) informations to files.
	 * 
	 * @param typeIdsFilename
	 *            the filename to the file contains the typeIds
	 * @param associationIdsFilename
	 *            the filename to the file contains the associationIds
	 * @throws IOException
	 */
	public void persistIDRelationsToFile(final String typeIdsFilename, final String associationIdsFilename)
			throws IOException {
		try (FileOutputStream file = new FileOutputStream(typeIdsFilename)) {
			this.typeNameIdRelation.store(file, "");
			file.flush();
		}
		
		try (FileOutputStream file = new FileOutputStream(associationIdsFilename)) {
			this.associationNameIdRelation.store(file, "");
			file.flush();
		}
	}
	
	/**
	 * This operation returns the next unused id for a new type.
	 * 
	 * @param typeName
	 *            is the name of the type for which a new id will be registered.
	 * @return the id
	 */
	public synchronized long pullNextUnusedTypeID(final String typeName) {
		if (typeName.equals(StringType.STRING_NAME) || typeName.equals(IntegerType.INTEGER_NAME)) {
			throw new IDContractViolationException();
		}
		
		this.currentlyUsedTypeMaxId += 1;
		this.typeNameIdRelation.setProperty(typeName, Long.toString(this.currentlyUsedTypeMaxId));
		return this.currentlyUsedTypeMaxId;
	}
	
	/**
	 * This operation returns the next unused id for a new association.
	 * 
	 * @param associationName
	 *            is the name of the association for which a new id will be registered.
	 * @return the id
	 */
	public synchronized long pullNextUnusedAssociationID(final String associationName) {
		this.currentlyUsedAssociationMaxId += 1;
		this.associationNameIdRelation.setProperty(associationName, Long.toString(this.currentlyUsedAssociationMaxId));
		return this.currentlyUsedAssociationMaxId;
	}
	
	/**
	 * This operations delivers the id for a type by the name of the parameter.
	 * 
	 * @param typeName
	 *            the name of the type to look for
	 * @return the Id of the type when found
	 * @exception IDNotFoundForNameException
	 *                is thrown if no id was found.
	 */
	public long findIdForType(final String typeName) throws IDNotFoundForNameException {
		if (!this.typeNameIdRelation.containsKey(typeName)) {
			throw new IDNotFoundForNameException();
		}
		return Long.parseLong(this.typeNameIdRelation.getProperty(typeName));
	}
	
	/**
	 * This operations delivers the id for a association by the name of the parameter.
	 * 
	 * @param associationName
	 *            the name of the association to look for
	 * @return the id of the association if found
	 * @exception IDNotFoundForNameException
	 *                is thrown if no id was found.
	 */
	public long findIdForAssociation(final String associationName) throws IDNotFoundForNameException {
		if (!this.associationNameIdRelation.containsKey(associationName)) {
			throw new IDNotFoundForNameException();
		}
		return Long.parseLong(this.associationNameIdRelation.getProperty(associationName));
	}
}
